# coding=utf-8
#
# Copyright (c) 2021 LIBZENT Innovations, Inc. BACKBONEstudio.
# Copyright (c) 2021 LIBZENT BIGFOOT Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Author:
#   LIBZENT Innovations, Inc. BACKBONEstudio
#   BIGFOOT Inc.
# date:
#   2021/03/30 v1.00 release
#   2021/04/06 v1.10 fix garbled character
#   2021/05/20 v1.20 fix toggleShowHUD button
#   2021/10/29 v1.30 add HUD of comment and fps
#   2022/08/23 v1.40 also available Python 2 and 3
#   2023/03/03 v1.41 added license terms and fixed bug

import maya.cmds as cmds
import maya.mel as mel
import os
import json
import copy
from maya.app.general import mayaMixin
from maya import OpenMayaUI
import sys
import inspect
import datetime
import math
import imp
try:
    imp.find_module('PySide2')
    import PySide2
    from PySide2.QtWidgets import *
    from PySide2.QtGui import *
    from PySide2.QtCore import *
    from PySide2.QtUiTools import *
    import shiboken2
    from shiboken2 import wrapInstance
except ImportError:
    import PySide
    from PySide.QtGui import *
    from PySide.QtCore import *
    from PySide.QtUiTools import *
    import shiboken
    from shiboken import wrapInstance

ptr = OpenMayaUI.MQtUtil.mainWindow()
try:
    parent = shiboken2.wrapInstance(int(ptr), QWidget)
except:
    parent = shiboken.wrapInstance(int(ptr), QWidget)
cw = QWidget()
camSpeedUnit = 'mph'
velSpeedUnit = 'mph'
distanceUnit = 'm'
velNodes = ['']*5
distanceNodes = ['']*5

path = os.path.abspath(inspect.getsourcefile(lambda:0))
path = os.path.join(os.path.normpath(os.path.join(path,'../')))
path = path.replace('\\','/')
if not dir in sys.path:
    sys.path.append(path)
import settings as s
imp.reload (s)


class GUI(mayaMixin.MayaQWidgetBaseMixin, QMainWindow, QFrame):
    buttonHeight = 16
    buttonWidth = 120
    blockNum = 6 #max:18
    defaultFileName = 'defaultSetting.json'
    defPath = '%s/%s'%(path, defaultFileName)
    baseDict = {}
    colorWindowName = 'HUD color settings'
    colorWindowObjName = 'HUDColorSettingsWin'

    labelColor = 16
    valueColor = 16
    
    def __init__(self, *args, **kwargs):
        super(GUI, self).__init__(*args, **kwargs)
        
        self.setWindowTitle("HUD custom tools")
        self.setObjectName("HUDCustomToolsUI")
        
        self.initBaseDict()
        self.importSetting()
                
        self.setAcceptDrops(True)
        self.createMenu()
        self.createUI()
        
        # Default Settings import
        if os.path.exists(self.defPath): 
            self.reloadFromFile(self.defPath)
    
    def allHUDvisOff(self):
        for hud in cmds.headsUpDisplay(q=1, listHeadsUpDisplays=1):
            cmds.headsUpDisplay(hud, vis=0, e=1)

    def openColorSettings(self):
        if cmds.window(self.colorWindowObjName, exists=True):
            cmds.deleteUI(self.colorWindowObjName, window=True)

        if cmds.windowPref(self.colorWindowObjName, exists=True):
            cmds.windowPref(self.colorWindowObjName, remove=True)
        
        cmds.window(self.colorWindowObjName, t = self.colorWindowName)
        cmds.columnLayout()
        self.labelColorSlider = cmds.colorIndexSliderGrp(
            label='HUD Label Color',
            min=2,
            max=32,
            value=self.labelColor+1,
            cc=self.changeLabelColorSlider,
            cl3=('center','left','left')
        )
        self.valueColorSlider = cmds.colorIndexSliderGrp(
            label='HUD Value Color',
            min=2,
            max=32,
            value=self.valueColor+1,
            cc=self.changeValueColorSlider,
            cl3=('center', 'left', 'left')
        )
        cmds.showWindow()

    def initBaseDict(self):
        sdict = {}
        for d in s.__dict__:
            try:
                if not '__' in d:
                    sdict[d] = s.__dict__[d]
            except:
                pass
    
        self.baseDict = copy.deepcopy(sdict)
        
    def createMenu(self):
        importAct = QAction('Import Settings', self)
        importAct.triggered.connect(self.importFile) 
        exportAct = QAction('Export Settings', self)
        exportAct.triggered.connect(self.exportFile)
        saveDefAct = QAction('Save As Default Settings', self)
        saveDefAct.triggered.connect(self.saveDef)
        initDefAct = QAction('Reset Settings', self)
        initDefAct.triggered.connect(self.initDef)
        visoffAct = QAction('All Visibility Off', self)
        visoffAct.triggered.connect(lambda: self.setVis(False))
        visonAct = QAction('All Visibility On', self)
        visonAct.triggered.connect(lambda: self.setVis(True))
        colorAct = QAction('Color Settings', self)
        colorAct.triggered.connect(lambda: self.openColorSettings())
        
        menuBar = self.menuBar()
        filemenu = menuBar.addMenu('File')
        filemenu.addAction(importAct)
        filemenu.addAction(exportAct) 
        filemenu.addSeparator()
        filemenu.addAction(saveDefAct)
        filemenu.addAction(initDefAct)
        editmenu = menuBar.addMenu('Edit')
        editmenu.addAction(visonAct)
        editmenu.addAction(visoffAct)
        editmenu.addSeparator()
        editmenu.addAction(colorAct)
    
    def saveDef(self):
        output = self.getExportJson()
        with open('%s/%s'%(path, self.defaultFileName), mode = 'w') as f:
            json.dump(output, f, indent=4)
                    
    def initDef(self):
        self.initBaseDict()
        if os.path.exists(self.defPath):
            ret = QMessageBox.information(
                None,
                'Confirm',
                'Delete default settings?',
                QMessageBox.Yes,
                QMessageBox.No
            )
            if ret == QMessageBox.Yes:
                os.remove(self.defPath)
        
        camSpeedCBIndex = self.camSpeedUnitCB.findText(camSpeedUnit)
        velSpeedCBIndex = self.velSpeedUnitCB.findText(velSpeedUnit)
        self.camSpeedUnitCB.setCurrentIndex(camSpeedCBIndex)
        self.velSpeedUnitCB.setCurrentIndex(velSpeedCBIndex)
        self.distanceCB.setCurrentIndex(self.distanceCB.findText(distanceUnit))
        self.fontsizeCB.setCurrentIndex(self.fontsizeCB.findText(self.HUDSize))
        self.labelColor = 16
        self.valueColor = 16
        
        for key in self.baseDict:
            val  = self.baseDict[key]
            if type(val) == dict:
                for i in range(5):
                    if val.get('name') == 'HUDDistance%s'%(i+1):
                        _t = val.get('node')
                        _l = val.get('label')
                        _l = _l.replace(' : ','')
                        self.distanceNameLEs[i].setText(_t)
                        self.distanceLabelLEs[i].setText(_l)
                    if val.get('name') == 'HUDVel%s'%(i+1):
                        _t = val.get('node')
                        _l = val.get('label')
                        _l = _l.replace(' : ','')
                        self.velocityNameLEs[i].setText(_t)
                        self.velocityLabelLEs[i].setText(_l)
                for i in range(3):
                    if val.get('name') == 'HUDComment%s'%(i+1):
                        _t = val.get('label')
                        self.commentLEs[i].setText(_t)
            
        self.reloadButtonLabels()
        
    def setVis(self, on):
        for b in self.buttons:
            b.setVis(on)
    
    def importFile(self):
        fpath = cmds.fileDialog2(
            fileFilter='*.json',
            dialogStyle=2, fm=1, dir=path
        )
        if fpath == None:
            return False        
        self.reloadFromFile(fpath[0])
    
    def reloadFromFile(self, filePath):
        with open(filePath, mode = 'r') as f:
            imports = json.load(f)
        for dKey in self.baseDict:
            d = self.baseDict[dKey]
            if type(d) == dict:
                for importName in imports:
                    if d.get('name') == importName:
                        importSettings = imports[importName]
                        for k in importSettings:
                            self.baseDict[dKey][k] = importSettings[k]
                        
                        for i in range(5):
                            if importName == 'HUDVel%s'%(i+1):
                                _t = importSettings.get('node')
                                _l = importSettings.get('label')
                                _l = _l.replace(' : ','')
                                self.velocityNameLEs[i].setText(_t)
                                self.velocityLabelLEs[i].setText(_l)
                            if importName == 'HUDDistance%s'%(i+1):
                                _t = importSettings.get('node')
                                _l = importSettings.get('label')
                                _l = _l.replace(' : ','')
                                self.distanceNameLEs[i].setText(_t)
                                self.distanceLabelLEs[i].setText(_l)
                        for i in range(3):
                            if importName == 'HUDComment%s'%(i+1):
                                _t = importSettings.get('label')
                                self.commentLEs[i].setText(_t)

        if 'unitOfCamSpeed' in imports:
            self.camSpeedUnitCB.setCurrentText(imports['unitOfCamSpeed'])
        if 'unitOfVelSpeed' in imports:
            self.velSpeedUnitCB.setCurrentText(imports['unitOfVelSpeed'])
        if 'unitOfDistance' in imports:
            self.distanceCB.setCurrentText(imports['unitOfDistance'])
        if 'HUDSizeNum' in imports:
            self.fontsizeCB.setCurrentIndex(imports['HUDSizeNum'])
        if 'HUDLabelColor' in imports:
            self.labelColor = imports.get('HUDLabelColor',16)
        if 'HUDValueColor' in imports:
            self.valueColor = imports.get('HUDValueColor',16)
            

        self.reloadButtonLabels()
        
    def exportFile(self):
        fpath = cmds.fileDialog2(
            fileFilter='*.json',
            dialogStyle=2, fm=0, dir=path
        )
        if fpath == None:
            return False
            
        output = self.getExportJson()
        
        with open(fpath[0], mode='w') as f:
            json.dump(output, f, indent=4)
            
    def getExportJson(self):
        output = {}
        output['unitOfCamSpeed'] = self.camSpeedUnitCB.currentText()
        output['unitOfVelSpeed'] = self.velSpeedUnitCB.currentText()
        output['unitOfDistance'] = self.distanceCB.currentText()
        output['HUDSizeNum'] = self.fontsizeCB.currentIndex()
        output['HUDLabelColor'] = self.labelColor
        output['HUDValueColor'] = self.valueColor
        for b in self.buttons:
            nam = b.setting.nam
            if nam: 
                output[nam] =  {
                    'section': b.setting.section, 
                    'block' : b.setting.block,
                    'vis' : b.setting.vis,
                    'node' : b.setting.node,
                    'label' : b.setting.lab,
                }
        return output
            
    def dragEnterEvent(self, event):
        super(GUI, self).dragEnterEvent(event)
        event.accept()
        
    def dragLeaveEvent(self, event):
        super(GUI, self).dragLeaveEvent(event)

        for hud in self.buttons:
            hud.releaseDrag()
        event.accept()
        
    def isLabelEnter(self):
        for hud in self.buttons:
            if hud.isEnter:
                return True
        return False
        
    def dropEvent(self, event):
        super(GUI, self).dropEvent(event)
        for hud in self.buttons:
            hud.releaseDrag()
        event.accept()
    
    def importSetting(self):
        self.dictSB={}
        self.HUDSize='small'
        
        doubleKey = []
        for setting in self.baseDict.keys():
            key = self.baseDict[setting]
            if type(key) == dict:
                if 'section' in key:
                    keyT = (key['section'], key['block'])
                    if keyT in self.dictSB:
                        doubleKey.append(self.dictSB[keyT])
                    self.dictSB[keyT] = setting
            else:
                if setting == 'HUDSizeNum': self.HUDSize = getHUDSize(key)
        for dk in doubleKey:
            for sec in range(10):
                for blk in range(self.blockNum):
                    keyT = (sec, blk)
                    if not keyT in self.dictSB:
                        self.dictSB[keyT] = dk
                        self.baseDict[dk]['section'] = sec
                        self.baseDict[dk]['block'] = blk
                        break
                else: continue
                break

        global camSpeedUnit
        camSpeedUnit = self.baseDict['unitOfCamSpeed']
        global velSpeedUnit
        velSpeedUnit = self.baseDict['unitOfVelSpeed']
        global distanceUnit
        distanceUnit = self.baseDict['unitOfDistance']
        if cmds.window(self.colorWindowObjName, exists=True):
            cmds.deleteUI(self.colorWindowObjName, window=True)

    def closeEvent(self, event):
        if cmds.window(self.colorWindowObjName, exists=True):
            cmds.deleteUI(self.colorWindowObjName, window=True)

        if cmds.windowPref(self.colorWindowObjName, exists=True):
            cmds.windowPref(self.colorWindowObjName, remove=True)

    def createUI(self):
        self.setCentralWidget(cw)
        
        lo = QVBoxLayout(cw)
        loU = QHBoxLayout()
        loD = QHBoxLayout()

        l = QLabel('-- HUD custom tools v1.41 --')
        l.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        l.setStyleSheet("QLabel{background-color : rgba(0,10,0,1)}")
        l.setFixedHeight(20)

        lo.addWidget(l)

        #buttons UI
        loUDFL = QHBoxLayout()
        loUDF = QmyFrame()
        
        loUDF.setFrameStyle(QFrame.Panel | QFrame.Raised)
        loUDF.setLineWidth(2)
        loUDF.setStyleSheet(
            "QmyFrame{background-image : url('%s/backgroundGUI.png')}"%path
        )
        loUD = QVBoxLayout()
        
        loUDF.setFixedSize(720,314)
        
        lo.addLayout(loUDFL)
        loUDFL.addWidget(loUDF)
        loUDF.setLayout(loUD)
        
        self.buttons=[]
        loUD.addLayout(loU)
        loU.addSpacing(10)
        for sec in range(5):
            loSection = QVBoxLayout()
            loSection.addSpacing(10)
            loU.addLayout(loSection)
            for blc in range(self.blockNum):
                dict,label = self.getButtonLabel(sec, blc)
                lab = DraggableLabel(label, setting=dict)
                lab.setFixedSize(self.buttonWidth, self.buttonHeight)
                loSection.addWidget(lab)
                self.buttons.append(lab)
        loM=QVBoxLayout()
        loUD.addLayout(loM)
        loM.addSpacing(40)

        loUD.addLayout(loD)
        loD.addSpacing(10)
        for sec in range(5,10):
            loSection = QVBoxLayout()
            loD.addLayout(loSection)
            for blc in range(self.blockNum-1, -1, -1):
                dict,label = self.getButtonLabel(sec, blc)
                lab = DraggableLabel(label, setting=dict)
                lab.setFixedSize(self.buttonWidth, self.buttonHeight)
                loSection.addWidget(lab)
                self.buttons.append(lab)
                
            loSection.addSpacing(10)

        #separate
        sep = QFrame()
        sep.setFrameShape( QFrame.HLine )
        sep.setFrameShadow( QFrame.Raised )
        sep.setSizePolicy(
            QSizePolicy.MinimumExpanding,
            QSizePolicy.MinimumExpanding
        )
        lo.addWidget(sep)

        f = QFrame()
        f.setStyleSheet("QFrame{background-color:#555555}")
        f.setFixedHeight(230)
        loS1 = QVBoxLayout()
        loS2 = QHBoxLayout()

        loS1.addLayout(loS2)

        # user name
        loS2.addStretch(27)
        l = QLabel('User Name:')
        l.setFixedWidth(65)
        loS2.addWidget(l)

        self.userLE = QLineEdit(eval(self.baseDict['userSettings']['command']))
        self.userLE.setFixedWidth(110)
        
        self.userLE.textEdited.connect(lambda: self.changeUsername())
        
        loS2.addWidget(self.userLE)
        loS2.addStretch(25)

        # unit of camSpeed
        l = QLabel('Unit of Velocity(Camera):')
        l.setFixedWidth(130)
        loS2.addWidget(l)
        
        self.camSpeedUnitCB = QComboBox()
        self.camSpeedUnitCB.addItem('m/s')
        self.camSpeedUnitCB.addItem('km/h')
        self.camSpeedUnitCB.addItem('mph')
        self.camSpeedUnitCB.setFixedWidth(70)
        camSpeedunitCBIndex = self.camSpeedUnitCB.findText(camSpeedUnit)
        self.camSpeedUnitCB.setCurrentIndex(camSpeedunitCBIndex)
        loS2.addWidget(self.camSpeedUnitCB)
        loS2.addStretch(33)        
                
        # font size
        l = QLabel('Font Size:')
        l.setFixedWidth(60)
        loS2.addWidget(l)
        
        self.fontsizeCB = QComboBox()
        self.fontsizeCB.addItem('small')
        self.fontsizeCB.addItem('large')
        self.fontsizeCB.setFixedWidth(90)
        fontsizeCBIndex = self.fontsizeCB.findText(self.HUDSize)
        self.fontsizeCB.setCurrentIndex(fontsizeCBIndex)
        loS2.addWidget(self.fontsizeCB)
        loS2.addStretch(31)
        
        # comments
        loS6 = QHBoxLayout()
        loS1.addLayout(loS6)
        comment1Lb = QLabel('Comment1: ')
        comment2Lb = QLabel('Comment2: ')
        comment3Lb = QLabel('Comment3: ')
        comment1Lb.setFixedWidth(65)
        comment2Lb.setFixedWidth(65)
        comment3Lb.setFixedWidth(65)
        self.commentLEs = []
        l = QLineEdit()
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 0:self.changeComment(_))
        self.commentLEs.append(l)
        l = QLineEdit()
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 1:self.changeComment(_))
        self.commentLEs.append(l)
        l = QLineEdit()
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 2:self.changeComment(_))
        self.commentLEs.append(l)

        
        loS6.addStretch(27)
        loS6.addWidget(comment1Lb)
        loS6.addWidget(self.commentLEs[0])
        loS6.addStretch(25)
        loS6.addWidget(comment2Lb)
        loS6.addWidget(self.commentLEs[1])
        loS6.addStretch(33)
        loS6.addWidget(comment3Lb)
        loS6.addWidget(self.commentLEs[2])
        loS6.addStretch(31)
    



        sep = QFrame()
        sep.setFrameShape( QFrame.HLine )
        sep.setFrameShadow( QFrame.Raised )
        sep.setSizePolicy(
            QSizePolicy.MinimumExpanding,
            QSizePolicy.MinimumExpanding
        )
        loS1.addWidget(sep)
        
        loS4 = QHBoxLayout()
        loS1.addLayout(loS4)
        
        loS3 = QVBoxLayout()
        
        loS4.addWidget(QLabel(''))
        loS4.addSpacing(10)
        loS4.addLayout(loS3)
        
        loS5 = QHBoxLayout()
        loS3.addLayout(loS5)
        
        # unit of velSpeed        
        l = QLabel('Unit of Velocity(Vel1-5):')
        l.setFixedWidth(130)
        loS5.addWidget(l)
        
        self.velSpeedUnitCB = QComboBox()
        self.velSpeedUnitCB.addItem('m/s')
        self.velSpeedUnitCB.addItem('km/h')
        self.velSpeedUnitCB.addItem('mph')
        self.velSpeedUnitCB.setFixedWidth(60)
        velSpeedUnitCBIndex = self.velSpeedUnitCB.findText(velSpeedUnit)
        self.velSpeedUnitCB.setCurrentIndex(velSpeedUnitCBIndex)
        loS5.addWidget(self.velSpeedUnitCB)
        loS5.addSpacing(50)
        
        self.velocityNameLEs = []
        self.velocityLabelLEs = []
        self.setVelocityPBs = []
        
        for i in range(5):
        # node vel
            loS5 = QHBoxLayout()
            l = QLabel('Velocity%s Node:'%(i+1))
            l.setFixedWidth(90)
            loS5.addWidget(l)

            l = QLineEdit()
            loS5.addWidget(l)
            l.textChanged.connect(lambda _,__=i: self.changeVelocityName(__))
            l.setFixedWidth(120)
            self.velocityNameLEs.append(l)

            l = QmyPushButton(self, '<<', 'velocity', i, width=20, layout=loS5)
            self.setVelocityPBs.append(l)
            
            
            vsLabel = self.baseDict['vel%sSettings'%(i+1)]['label']
            l=QLineEdit(vsLabel.replace(' : ',''))
            self.velocityLabelLEs.append(l)
            
            loS3.addLayout(loS5)
        
        
        loS4.addSpacing(100)
        loS3 = QVBoxLayout()
        
        loS4.addWidget(QLabel(''))
        loS4.addLayout(loS3)
        
        loS5 = QHBoxLayout()
        loS3.addLayout(loS5)
        
        # unit of distance        
        l = QLabel('Unit of Distance(Dist1-5):')
        l.setFixedWidth(130)
        loS5.addWidget(l)
        
        self.distanceCB = QComboBox()
        self.distanceCB.addItem('m')
        self.distanceCB.addItem('ft')
        self.distanceCB.setFixedWidth(60)
        self.distanceCB.setCurrentIndex(self.distanceCB.findText(distanceUnit))
        loS5.addWidget(self.distanceCB)
        loS5.addSpacing(50)
        
        self.distanceNameLEs = []
        self.distanceLabelLEs = []
        self.setDistancePBs = []
        
        ##distance
        for i in range(5):
            loS5 = QHBoxLayout()
            l = QLabel('Distance%s Node:'%(i+1))
            l.setFixedWidth(90)
            loS5.addWidget(l)

            l = QLineEdit()
            loS5.addWidget(l)
            l.textChanged.connect(lambda s, _ = i:self.changeDistanceName(_))
            l.setFixedWidth(120)
            self.distanceNameLEs.append(l)
            
            l = QmyPushButton(self, '<<', 'distance', i, width=20, layout=loS5)
            self.setDistancePBs.append(l)
            
            dsLabel = self.baseDict['distance%sSettings'%(i+1)]['label']
            l = QLineEdit(dsLabel.replace(' : ',''))
            self.distanceLabelLEs.append(l)
            
            loS3.addLayout(loS5)
        
        loS4.addWidget(QLabel(''))
        loS4.addWidget(QLabel(''))
        
        f.setLayout(loS1)
        lo.addWidget(f)
        lo.addSpacing(10)

        # exec button
        loB = QHBoxLayout()
        toggleShowHUDButton = QPushButton('Toggle [Show] > [HUD]')
        toggleShowHUDButton.clicked.connect(lambda: self.toggleShowHUD())
        toggleShowHUDButton.setFixedWidth(180)
        toggleShowHUDButton.setStyleSheet(
            "QPushButton{background-color : #335533}"
        )
        loB.addWidget(toggleShowHUDButton)
        showHUDButton = QPushButton('Set/Update HUD')
        showHUDButton.clicked.connect(lambda: self.showHUD())
        showHUDButton.setFixedWidth(320)
        showHUDButton.setStyleSheet(
            "QPushButton{background-color : #557755}"
        )
        loB.addWidget(showHUDButton)
        loB.setAlignment(Qt.AlignHCenter)
        lo.addLayout(loB)

    def toggleShowHUD(self,toggle=-1):
        active_panel = cmds.playblast(activeEditor=True)
        on = toggle
        if toggle == -1:
            on = not cmds.modelEditor(active_panel, q=1, hud=1)
        cmds.modelEditor(active_panel, edit=1, hud=on)
        cmds.optionVar(iv=("playblastShowHUD",on))
        mel.eval("updatePlayblastMenus\
        (\"playblastShowHUD\", \"showHUDItemPB\")")
        
    def setLw(self, b, label):
        length = len(label)
        b.setting.lw = 100
        if length > 13:
            b.setting.lw = length*8
        
    def setVelocityName(self, i):
        if self.velocityNameLEs[i].text() == '':
            sl = cmds.ls(sl=1)
            if not sl:
                cmds.confirmDialog(
                    title = 'Warning',
                    message = 'Select a node before pushing [<<] buttoon.',
                    button = ['OK']
                )
                return
            self.velocityNameLEs[i].setText(sl[0])
        else:
            self.velocityNameLEs[i].setText('')
    
    def setDistanceName(self, i):
        if self.distanceNameLEs[i].text() == '':
            sl = cmds.ls(sl=1)
            if not sl:
                cmds.confirmDialog(
                    title = 'Warning',
                    message = 'Select a node before pushing [<<] button.',
                    button = ['OK']
                )
                return
            self.distanceNameLEs[i].setText(sl[0])
        else:
            self.distanceNameLEs[i].setText('')
                
    def changeLabelColorSlider(self):
        idx = cmds.colorIndexSliderGrp(self.labelColorSlider, q=1, v=1) - 1
        self.labelColor = idx

    def changeValueColorSlider(self):
        idx = cmds.colorIndexSliderGrp(self.valueColorSlider, q=1, v=1) - 1
        self.valueColor = idx
        
    def setHUDLabelColor(self, col=-1):
        if col == -1:
            col = self.labelColor
        cmds.displayColor("headsUpDisplayLabels", col, dormant=1)
        
    def setHUDValueColor(self, col=-1):
        if col == -1:
            col = self.valueColor        
        cmds.displayColor("headsUpDisplayValues", col, dormant=1)
        
    def changeDistanceName(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDDistance%s'%(i+1):
                b.setting.node = self.distanceNameLEs[i].text()
                lab = 'Dist%s %s'%(i+1, b.setting.node)
                
                b.setVis(0 if  self.distanceNameLEs[i].text() == '' else 1)
                b.setting.lab = '%s : '%lab
                b.setText(lab)
                self.setLw(b, b.setting.lab)
        
    def changeDistanceLabel(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDDistance%s'%(i+1):
                b.setting.lab = "%s : "%self.distanceLabelLEs[i].text()
                b.setText(self.distanceLabelLEs[i].text())
        
                self.setLw(b, b.setting.lab)
                
    def changeUsername(self):
        for b in self.buttons:
            if b.setting.nam == 'HUDUser':
                b.setting.cmd = "'%s'"%self.userLE.text()
                b.setText(self.userLE.text())
    
    def changeComment(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDComment%s'%(i+1):
                b.setting.lab = '%s'%self.commentLEs[i].text()
                            
    def changeVelocityName(self,i):
        for b in self.buttons:
            if b.setting.nam == 'HUDVel%s'%(i+1):
                b.setting.node = self.velocityNameLEs[i].text()
                lab = 'Vel%s %s'%(i+1, b.setting.node)
                
                b.setVis(0 if  self.velocityNameLEs[i].text() == '' else 1)
                b.setting.lab = '%s : '%lab
                b.setText(lab)
                self.setLw(b, b.setting.lab)
                
    def changeVelocityLabel(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDVel%s'%(i+1):
                b.setting.lab = "%s : "%self.velocityLabelLEs[i].text()
                b.setText(self.velocityLabelLEs[i].text())
        
                self.setLw(b, b.setting.lab)
                    
    def showHUD(self):
        global camSpeedUnit
        camSpeedUnit = self.camSpeedUnitCB.currentText()
        global velSpeedUnit
        velSpeedUnit = self.velSpeedUnitCB.currentText()
        global distanceUnit
        distanceUnit = self.distanceCB.currentText()
        global distanceNodes
        global velNodes
        for i in range(5):
            distanceNodes[i] = self.distanceNameLEs[i].text()
            velNodes[i] = self.velocityNameLEs[i].text()

        self.allHUDvisOff()
        self.setHUDLabelColor()
        self.setHUDValueColor()
        self.toggleShowHUD(1)
        for b in self.buttons:
            b.setting.size = self.fontsizeCB.currentText()
            b.setting.setHUD()
            
    def getButtonLabel(self, sec, blc):
        dict = {}
        name = ''
        label = ''
        dict['section'] = sec
        dict['block'] = blc
        if (sec, blc) in self.dictSB:
            dict = self.baseDict[self.dictSB[(sec, blc)]]
            name = dict['name'].replace('HUD','')
            try :
                label = eval( dict['command'] )
            except:
                pass
            
            if 'label' in dict:
                label = dict['label'].replace(':','')
            if not label:
                label = name
            
            if label == 'Time':
                label = eval( dict['command'] )
            if name == 'CamName':
                label = 'CamName'
            for i in range(1,4):
                if name == 'Comment%s'%i:
                    label = 'Comment%s'%i
            
        dict['size'] = self.HUDSize
        return dict,label
    
    def reloadButtonLabels(self):
        self.importSetting()
        for sec in range(10):
            for blk in range(self.blockNum):
                blk_ = blk if sec < 5 else self.blockNum-blk-1
                dict,label = self.getButtonLabel(sec, blk_)
                self.buttons[sec*self.blockNum + blk].setting.reset(dict)
                self.buttons[sec*self.blockNum + blk].setText(label)
                self.buttons[sec*self.blockNum + blk].setVis(
                    self.buttons[sec*self.blockNum + blk].setting.vis
                )
        self.changeUsername()
        for i in range(5):
            self.changeDistanceLabel(i)
            self.changeVelocityLabel(i)


class QmyPushButton():
    def __init__(self, gui, text, VD, i, width, layout):
        self.num = i
        self.gui = gui
        self.button = QPushButton(text)
        self.button.setFixedWidth(width)
        layout.addWidget(self.button)
        self.VD = VD
        self.button.clicked.connect(self.clicked)
        
    def clicked(self):
        if(self.VD == 'velocity'):
            self.gui.setVelocityName(self.num)
            
        if(self.VD == 'distance'):
            self.gui.setDistanceName(self.num)


class QmyFrame(QFrame):
    def __init__ (self,*args,**kwargs):
        super(QmyFrame, self).__init__(*args, **kwargs)

            
class HUDsetting():
    def __init__(self,dict):
        self.reset(dict)
    def reset(self,dict):
        self.section = dict.get('section', '')
        self.block = dict.get('block', '')
        self.nam = dict.get('name', '')
        self.lab = dict.get('label', '')
        self.cmd = dict.get('command', '')
        self.evt = dict.get('event', '')
        self.lw = dict.get('lw', 80)
        self.dw = dict.get('dw', 30)
        self.ba = dict.get('ba', 'left')
        self.da = dict.get('da', 'left')
        self.size = dict.get('size', 'small')
        self.vis = dict.get('vis', 1 if self.nam else 0)
        self.node = dict.get('node','')
    
    def setHUD(self):    
        self.deleteHUD(self.nam, self.section, self.block)
        kwargs = {
            'section' : self.section, 
            'block' : self.block,
            'labelWidth' : self.lw,
            'dataWidth' : self.dw,
            'label' : self.lab,
            'blockAlignment' : self.ba,
            'dataAlignment' : self.da,
            'labelFontSize' : self.size,
            'dataFontSize' : self.size,
        }
        
        if self.vis:
            if self.cmd == '':
                cmds.headsUpDisplay(self.nam,  **kwargs)
            else:
                if self.evt == '':
                    cmds.headsUpDisplay(
                        self.nam, command=self.cmd, atr=1,  **kwargs
                    )
                else:
                    cmds.headsUpDisplay(
                        self.nam, command=self.cmd, event=self.evt, **kwargs
                    )

    def deleteHUD(self, nam, section, block):
        for h in cmds.headsUpDisplay(q=1, listHeadsUpDisplays=1):
            s = cmds.headsUpDisplay(h, q=1, section=1)
            b = cmds.headsUpDisplay(h, q=1, block=1)
            if ((s,b) == (section,block)) or (h == nam): 
                cmds.headsUpDisplay(h, rem=1)


class DraggableLabel(QLabel):
    isEnter = False
    startPos = QPoint(0, 0)
    def __init__(self, *args, **kwargs):
        dict = kwargs.pop('setting')
        self.setting = HUDsetting(dict)
        
        super(DraggableLabel, self).__init__(*args, **kwargs)
        
        self.setAcceptDrops(True)
        self.backgroundColor(0)
        self.drag = None

    def mousePressEvent(self, event):
        super(DraggableLabel, self).mousePressEvent(event)
        if event.button() == Qt.LeftButton:
            self.startPos = event.pos()
            event.accept()

    def mouseMoveEvent(self, event):
        distance = (event.pos() - self.startPos).manhattanLength()
        if distance > QApplication.startDragDistance():
            mimeData = QMimeData()
            mimeData.setText(self.text())
            
            self.backgroundColor(2)
            
            self.drag = QDrag(self)
            self.drag.setMimeData(mimeData)
            self.drag.exec_()
            
        event.accept()

    def dropEvent(self, event):
        super(DraggableLabel, self).dropEvent(event)
        if event.mimeData().hasText():
            event.source().backgroundColor(0)
            event.source().drag = None
            
            #flip text
            tmp = event.source().text()
            event.source().setText(self.text())
            self.setText(tmp)

            #flip setting
            tmp = event.source().setting
            event.source().setting = self.setting
            self.setting = tmp

            #nonflip section,block
            tmp = event.source().setting.section
            event.source().setting.section = self.setting.section
            self.setting.section = tmp
            tmp = event.source().setting.block
            event.source().setting.block = self.setting.block
            self.setting.block = tmp
            
            event.source().backgroundColor(0)
            
        self.backgroundColor(0)
        self.drag = None
        event.accept()
    
    def setVis(self, on):
        if self.setting.nam:
            self.setting.vis = on
        self.backgroundColor(0)
        
    def mouseReleaseEvent(self, event):
        super(DraggableLabel, self).mouseReleaseEvent(event)
        if self.setting.nam:
            self.setVis(1 - self.setting.vis)
        event.accept()

    def releaseDrag(self):
        self.backgroundColor(0)
        self.drag = None

    def dragLeaveEvent(self, event):
        super(DraggableLabel, self).dragLeaveEvent(event)
        if not self.drag:
            self.backgroundColor(0)

        event.accept()
        
    def dragEnterEvent(self, event):
        super(DraggableLabel, self).dragEnterEvent(event)
        if type(event.source()) == type(self):
            if not self.drag:
                self.backgroundColor(1)
                event.source().backgroundColor(2)
            event.accept()
#            event.acceptProposedAction()
        else:
            event.ignore()

    def backgroundColor(self,i):
        alpha = 0.75 if self.setting.vis else 0.25
        colors = [
            'rgba(0,70,70,%s)'%alpha,
            'rgba(0,120,120,%s)'%alpha,
            'rgba(60,60,0,%s)'%alpha,
            'rgba(0,100,100,%s)'%alpha
        ]
        self.setStyleSheet('QLabel { background-color :%s}'%colors[i])
    
    def enterEvent(self, event):
        super(DraggableLabel, self).enterEvent(event)
        isEnter = True
        self.backgroundColor(3)
        event.accept()
    
    def leaveEvent(self, event):
        super(DraggableLabel, self).enterEvent(event)
        isEnter = False
        if not self.drag:
            self.backgroundColor(0)
        event.accept()


def getDistance(i):
    i-=1

    try:
        if cmds.objExists(distanceNodes[i]):
            cmds.getAttr('%s.wm'%distanceNodes[i])
        else: return '---'
    except:
        return '---'
    camPos = convertLinearMag(cmds.getAttr('%s.wm'%getCam())[12:15])
    nodePos = convertLinearMag(cmds.getAttr('%s.wm'%distanceNodes[i])[12:15])
    
    norm = getL2norm([camPos[j] - nodePos[j] for j in range(3)])
    
    distance = cmds.convertUnit(norm,toUnit=distanceUnit)
    if distance.endswith(distanceUnit):
        distance = distance.replace(distanceUnit,'')
    norm = float(distance)
    
    return '{:.2f} {}'.format(norm, distanceUnit)

def convertLinearMag(v):
    if type(v) in (float,int):
        return v*float(cmds.convertUnit(1,fromUnit='cm'))
    if type(v) == list:
        return [convertLinearMag(_v) for _v in v]
    
def getHUDSize(i):
    return ['small', 'large'][i]
    
def getCam():
    for pane in mel.eval('paneLayout -q -ca $gMainPane'):
        panelName = cmds.getPanel(containing=pane)
        if not cmds.control(pane, q=1, io=1):
            cam = cmds.modelPanel(
                cmds.modelPanel(panelName, q=1, modelEditor=1), q=1, camera=1
            )
            typ = cmds.nodeType(cam)
            if typ == 'transform':
                return cam
            elif typ == 'camera':
                return cmds.listRelatives(cam, p=1)[0]
            else:
                cmds.error('camera type error.')

def getL2norm(x):
    return math.sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2])
    
def getNodeVel(i):
    node = velNodes[i-1]
    return getVel(node, velSpeedUnit)
    
def getCamVel():
    return getVel(getCam(), camSpeedUnit)
    
def getVel(node, unit):
    try:
        if cmds.objExists(node):
            cmds.getAttr('%s.wm'%node)
        else: return '---'
    except:
        return '---'
        
    pos_  = cmds.getAttr('%s.wm'%node)[12:15]
    pos = convertLinearMag(pos_)
    prePos_ = cmds.getAttr('%s.wm'%node, t=cmds.currentTime(q=1)-1)[12:15]
    prePos = convertLinearMag(prePos_)
    distance = getL2norm([pos[i]-prePos[i] for i in range(3)])

    distanceUnit, timeUnit = {
        'mph' : ('mi', 60*60), 
        'km/h' : ('km', 60*60),
        'm/s' : ('m', 1)
    }[unit]
    
    distance = cmds.convertUnit(distance, toUnit=distanceUnit)
    if distance.endswith(distanceUnit):
        distance = distance.replace(distanceUnit, '')
    distance = float(distance)
    
    vel = distance * mel.eval('currentTimeUnitToFPS') * timeUnit
    return '{:.2f} {}'.format(vel, unit)

def getFPS():
    ret = ''
    fps = mel.eval('currentTimeUnitToFPS')
    if fps.is_integer():
        ret = '{:.0f} FPS'.format(fps)
    else:
	    ret = '{:.3f} FPS'.format(fps) 
    return ret

def createWindow():
    ui = GUI()
    wid = QApplication.topLevelWidgets()
    for i in wid:
        if ui.objectName() == i.objectName():
            i.close()

    ui.setAttribute(Qt.WA_DeleteOnClose)
    ui.show()
    return ui

def oneShot():
    ui = GUI()
    ui.showHUD()
    
if __name__ == '__main__':
    createWindow()
