# coding=utf-8
#
# Copyright (c) 2021 LIBZENT Innovations, Inc. BACKBONEstudio.
# Copyright (c) 2021 LIBZENT BIGFOOT Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Author:
#   LIBZENT Innovations, Inc. BACKBONEstudio
#   BIGFOOT Inc.
# date:
#   2021/09/30 v1.00 release
#   2021/10/29 v1.10 appended [File Name Suffix] and [Use Audio]
#   2022/03/09 v1.11 fixed default start-end time
#   2022/08/23 v1.20 also available Python 2 and 3
#   2023/01/31 v1.21 fixed playblast params in Maya2023
#   2023/03/03 v1.22 added license terms and fixed bug

import maya.cmds as cmds
import maya.mel as mel
import os
import json
import copy
from maya.app.general import mayaMixin
from maya import OpenMayaUI
import sys
import inspect
import datetime
import math
import imp
try:
    imp.find_module('PySide2')
    import PySide2
    from PySide2.QtWidgets import *
    from PySide2.QtGui import *
    from PySide2.QtCore import *
    from PySide2.QtUiTools import *
    import shiboken2
    from shiboken2 import wrapInstance
except ImportError:
    import PySide
    from PySide.QtGui import *
    from PySide.QtCore import *
    from PySide.QtUiTools import *
    import shiboken
    from shiboken import wrapInstance

ptr = OpenMayaUI.MQtUtil.mainWindow()
try:
    parent = shiboken2.wrapInstance(int(ptr), QWidget)
except:
    parent = shiboken.wrapInstance(int(ptr), QWidget)
cw = QWidget()

path = os.path.abspath(inspect.getsourcefile(lambda:0))
path = os.path.join(os.path.normpath(os.path.join(path,'../')))
path = path.replace('\\','/')

version = 'v1.22'

class GUI(mayaMixin.MayaQWidgetBaseMixin, QMainWindow, QFrame):
    defaultFileName = 'defaultSetting.json'
    defPath = '%s/%s'%(path, defaultFileName)
    convAll = ['SCENE', 'CUT', '*', 'START', 'END', 'WIDTH', 'HEIGHT', 'FPS', 'OVERSCAN', 'VERSION', 'USER']
    initConv = ['SCENE', 'CUT', '*', 'START', 'END']
    conv = initConv
    rowData=[]
    
    CONVBOX_MAXNUM = 12 
    COLUMN_WIDTH_ITEM = 50
    COLUMN_WIDTH_OUTPUT = 60
    RGB_OK = (48,96,48)
    RGB_NG = (128,0,0)
    WINDOW_HEIGHT_MAX = 600

    DEFAULT_WINDOW_SIZE = (800,300)
    WINDOW_WIDTH_OPTIONVAL = 'playBlastToolWindowWidth'
    WINDOW_HEIGHT_OPTIONVAL = 'playBlastToolWindowHeight'
    DIRNAME_OPTIONVAL = 'playBlastToolDirName'
    
    mel.eval("if ( !`exists doPlayblastArgList`) { source doPlayblastArgList.mel;}")

    def __init__(self, *args, **kwargs):
        super(GUI, self).__init__(*args, **kwargs)

        self.setWindowTitle('playblast tool')
        self.setObjectName('playblastToolUI')
        self.setWindowSize()

        self.createMenu()
        self.createUI()

        # Default Settings import
        if os.path.exists(self.defPath): 
            self.reloadFromFile(self.defPath)

        self.updateConv()
    
    def setWindowSize(self):
        if cmds.optionVar(ex = self.WINDOW_WIDTH_OPTIONVAL):
            self.resize(cmds.optionVar(q=self.WINDOW_WIDTH_OPTIONVAL),
                        cmds.optionVar(q=self.WINDOW_HEIGHT_OPTIONVAL))
        else:
            self.resize(*self.DEFAULT_WINDOW_SIZE)
    
    def resizeEvent(self, event):
        cmds.optionVar(intValue=
            [self.WINDOW_WIDTH_OPTIONVAL, self.size().width()]
        )
        cmds.optionVar(intValue=
            [self.WINDOW_HEIGHT_OPTIONVAL, self.size().height()]
        )

    def remSizeOptionVal(self):
        cmds.optionVar(remove = self.WINDOW_WIDTH_OPTIONVAL)
        cmds.optionVar(remove = self.WINDOW_WIDTH_OPTIONVAL)
        
    def createMenu(self):
        importAct = QAction('Import Settings',self)
        importAct.triggered.connect(self.importFile)
        exportAct = QAction('Export Settings',self)
        exportAct.triggered.connect(self.exportFile)
        saveDefAct = QAction('Save As Default Settings', self)
        saveDefAct.triggered.connect(self.saveDef)
        initDefAct = QAction('Reset Settings', self)
        initDefAct.triggered.connect(self.initDef)
        
        outputOffAct = QAction('All Output Off', self)
        outputOffAct.triggered.connect(lambda: self.setAllOutput(False))
        outputOnAct = QAction('All Output On', self)
        outputOnAct.triggered.connect(lambda: self.setAllOutput(True))
        
        menuBar = self.menuBar()
        filemenu = menuBar.addMenu('File')
        filemenu.addAction(importAct)
        filemenu.addAction(exportAct)
        filemenu.addSeparator()
        filemenu.addAction(saveDefAct)
        filemenu.addAction(initDefAct)
        """
        editmenu = menuBar.addMenu('Edit')
        editmenu.addAction(outputOffAct)
        editmenu.addAction(outputOnAct)
        """
    
    def importFile(self):
        fpath = cmds.fileDialog2(
            fileFilter='*.json',
            dialogStyle=2, fm=1, dir=path
        )
        if fpath == None:
            return False
        self.reloadFromFile(fpath[0])

    def reloadFromFile(self, path):
        importData = []
        with open(path, mode = 'r') as f:
            importData = json.load(f)
        self.conv = importData['convention']
        self.convBoxes.reload()

        # UI Update
        if 'calcStartFrameCheck' in importData:
            self.csfCb.setChecked(importData['calcStartFrameCheck'])
        if 'calcStartFrame' in importData:
            self.csfLe.setText(importData['calcStartFrame'])
        if 'useAudioCheck' in importData:
            self.audioCb.setChecked(importData['useAudioCheck'])
        if 'saveDirectory' in importData:
            self.dirLe.setText(importData['saveDirectory'])

    def exportFile(self):
        fpath = cmds.fileDialog2(
            fileFilter='*.json',
            dialogStyle=2, fm=0, dir=path
        )
        if fpath == None:
            return False
        
        self.exportFileFromPath(fpath[0])

    def saveDef(self):
        self.exportFileFromPath('%s/%s'%(path, self.defaultFileName))

    def exportFileFromPath(self, path):
        output = {
            'convention':self.conv,
            'calcStartFrameCheck':self.csfCb.isChecked(),
            'calcStartFrame':self.csfLe.text(),
            'useAudioCheck':self.audioCb.isChecked(),
            'saveDirectory':self.dirLe.text()
        }
        
        with open(path, mode = 'w') as f:
            json.dump(output, f, indent=4)

    def initDef(self):
        if os.path.exists(self.defPath):
            ret = QMessageBox.information(
                None,
                'Confirm',
                'Delete default settings?',
                QMessageBox.Yes,
                QMessageBox.No
            )
            if ret == QMessageBox.Yes:
                os.remove(self.defPath)
        self.conv = self.initConv
        self.convBoxes.reload()


    class conventionsBox:
        def __init__(self, gui, num, lo):
            self.lo = lo
            self.gui = gui
            self.num = num

            self.setMenu()
            self.reloadMenu()

        class QmyAction:
            def __init__(self, label, convBox):
                self.convBox = convBox
                self.action = QAction(label, self.convBox.gui)
                self.action.triggered.connect(lambda : self.changeLabel(label))
                self.convBox.mn.addAction(self.action)
            def changeLabel(self, label):
                self.convBox.tb.setText(label + '    ')
                #self.convBox.tb.text().setAlignment(Qt.AlignLeft)
                self.convBox.gui.conv[self.convBox.num] = label


        def setMenu(self):
            self.tb = QToolButton()
            self.mn = QMenu(self.gui)

            self.tb.setMenu(self.mn)
            
            self.tb.setPopupMode(QToolButton.InstantPopup)

            self.tb.setStyleSheet('border: 1px;\
                                   background-color: rgba(10,80,60,1);')
            self.tb.setFixedHeight(20)
            self.lo.addWidget(self.tb,
                alignment=(Qt.AlignLeft| Qt.AlignVCenter)
            )

        def addConvLabel(self):
            self.mn.addSeparator()
            for r in self.gui.convAll:
                self.QmyAction(r, self)
            
        def addRemLabel(self):
            trashIcon = QApplication.style().standardIcon(
                QStyle.SP_DialogDiscardButton
            )
            removeAc = QAction (trashIcon, 'remove', self.gui)
            removeAc.triggered.connect(
                lambda: self.gui.convBoxes.remove(self.num)
            )
            self.mn.addAction(removeAc)                

        def addInsLabel(self):
            insertIcon = QApplication.style().standardIcon(
                QStyle.SP_FileDialogBack
            )
            insertAc = QAction (insertIcon, 'insert', self.gui)
            insertAc.triggered.connect(
                lambda: self.gui.convBoxes.insert(self.num)
            )
            self.mn.addAction(insertAc)

        def addBlankLabel(self):
            ac = QAction(" ")
            self.mn.addAction(ac)

        def reloadMenu(self):
            self.mn.clear()
            if self.num < len(self.gui.conv):
                if len(self.gui.conv) < self.gui.CONVBOX_MAXNUM:
                    self.addInsLabel()
                self.addRemLabel()
                self.addConvLabel()
                self.tb.setText(self.gui.conv[self.num] + "    ")
            """
            if self.num == len(self.gui.conv):
                if len(self.gui.conv) < self.gui.CONVBOX_MAXNUM:
                    self.addInsLabel()
                self.tb.setText('...    ')            
            """
            if self.num >= len(self.gui.conv):
                self.tb.hide()
            else:
                self.tb.show()
    
    class conventionsBoxes:
        def __init__(self, gui, lo):
            self.gui = gui
            self.lo = lo
            self.convBoxes=[]
            for i in range(self.gui.CONVBOX_MAXNUM):
                self.convBoxes.append(
                    self.gui.conventionsBox(self.gui, i, self.lo)
                )
            
            self.plusPb = QPushButton(' + ')
            self.plusPb. setStyleSheet(
                'QPushButton{background-color : rgba(10,80,60,1)}'
            )
            self.plusPb.setFixedHeight(20)
            self.plusPb.clicked.connect(lambda: self.insert(len(self.gui.conv)))
            self.lo.addWidget(self.plusPb)
            self.plusPbShow()

        def insert(self, n):
            self.gui.conv.insert(n,self.gui.convAll[2])
            self.reload()

        def remove(self, n):
            self.gui.conv.pop(n)
            self.reload()

        def reload(self):
            for rb in self.convBoxes:
                rb.reloadMenu()
            self.plusPbShow()

        def plusPbShow(self):
            if len(self.gui.conv) < self.gui.CONVBOX_MAXNUM:
                self.plusPb.show()
            else:
                self.plusPb.hide()
        
    def createUI(self):
        self.setCentralWidget(cw)
        lo = QVBoxLayout(cw)
        
        #title
        l = QLabel('-- playblast tool %s --'%version)
        l.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        l.setStyleSheet('QLabel{background-color : rgba(0,10,5,1)}')
        l.setFixedHeight(20)

        lo.addWidget(l)

        #conv
        loConv = QHBoxLayout()
        loConvHA = QHBoxLayout()
        loConvHB = QHBoxLayout()
        loFr = QFrame()
        loFr.setLayout(loConv)
        lo.addWidget(loFr)
        loConv.addLayout(loConvHA)
        loConv.addLayout(loConvHB)

        loFr.setFrameStyle(QFrame.Panel | QFrame.Raised)
        loFr.setLineWidth(2)
        loFr.setAutoFillBackground(True)
        loFr.setStyleSheet('background-color: rgba(25,50,50,0.5)')

        lb = QLabel(' Cam Name: ')
        loConvHA.addWidget(lb)
        lb.setStyleSheet('QLabel{background-color: rgba(0,0,0,0)}')
        self.convBoxes = self.conventionsBoxes(self, loConvHA)
        loConvHA.addStretch(0)
        updateConvPb = QPushButton('Update')
        updateConvPb.setFixedWidth(70)
        updateConvPb.clicked.connect(lambda:self.updateConv(self.getNowCams()))
        updateConvPb.setStyleSheet(
            'QPushButton{background-color : rgba(20,100,80,1)}'
        )
        loConvHB.addWidget(updateConvPb)


        lo.addSpacing(10)
        
        #separate
        sep = QFrame()
        sep.setFrameShape( QFrame.HLine )
        sep.setFrameShadow( QFrame.Raised )
        lo.addWidget(sep)

        #cam buttons
        loCam = QHBoxLayout()
        setCamPb = QPushButton('Set Selected Cams')
        addCamPb = QPushButton('Add Selected Cams')
        remCamPb = QPushButton('Remove Selected Cams')
        selCamPb = QPushButton('Select all Cams')
        outputPb = QPushButton('All Output ON/OFF')
        setCamPb.setFixedWidth(140)
        addCamPb.setFixedWidth(140)
        remCamPb.setFixedWidth(140)
        selCamPb.setFixedWidth(120)
        outputPb.setFixedWidth(120)
        setCamPb.clicked.connect(self.setCams)
        addCamPb.clicked.connect(self.addCams)
        remCamPb.clicked.connect(self.remCams)
        selCamPb.clicked.connect(self.selectNowCams)
        outputPb.clicked.connect(lambda:self.setAllOutput(-1))
        loCam.addWidget(setCamPb)
        loCam.addWidget(addCamPb)
        loCam.addWidget(remCamPb)
        loCam.addStretch(0)
        loCam.addWidget(selCamPb)
        lo.addLayout(loCam)
        loCam.addStretch(0)
        loCam.addWidget(outputPb)

        #table
        self.tableItemEditable = False
        self.tbl = QTableWidget()
        lo.addWidget(self.tbl)
        self.tbl.itemChanged.connect(self.changeItem)
        self.tbl.itemSelectionChanged.connect(self.selectItem)
        
        defaultDir = ''
        if cmds.optionVar(ex = self.DIRNAME_OPTIONVAL):
            defaultDir = cmds.optionVar(q = self.DIRNAME_OPTIONVAL)
        
        #setting
        pbSettingLo1 = QHBoxLayout()
        osLb = QLabel('Time Range Offset: ')
        self.offsetLe = QLineEdit('0')
        m12Pb = QPushButton('-12')
        p12Pb = QPushButton('+12')
        self.appPb = QPushButton('Rename Apply')
        self.appPb.setEnabled(False)
        m12Pb.clicked.connect(lambda: self.offsetPlusMinus(-12))
        p12Pb.clicked.connect(lambda: self.offsetPlusMinus(12))
        self.appPb.clicked.connect(self.renameApply)
        osLb.setFixedWidth(100)
        self.offsetLe.setFixedWidth(40)
        m12Pb.setFixedWidth(32)
        p12Pb.setFixedWidth(32)
        self.appPb.setFixedWidth(100)
        pbSettingLo1.addWidget(osLb)
        pbSettingLo1.addWidget(self.offsetLe)
        pbSettingLo1.addWidget(m12Pb)
        pbSettingLo1.addWidget(p12Pb)
        pbSettingLo1.addWidget(self.appPb)
        pbSettingLo1.addSpacing(50)
        
        self.csfCb = QCheckBox()
        self.csfCb.setChecked(0)
        csfLb = QLabel('Calc Start Frame: ')
        self.csfLe = QLineEdit(str(cmds.playbackOptions(ast=1,q=1)))
        self.csfLe.setEnabled(0)
        csfLb.setFixedWidth(90)
        self.csfLe.setFixedWidth(40)
        pbSettingLo1.addWidget(self.csfCb)
        pbSettingLo1.addWidget(csfLb)
        pbSettingLo1.addWidget(self.csfLe)
        pbSettingLo1.addStretch(0)      
        
        self.csfCb.stateChanged.connect(
            lambda:self.csfLe.setEnabled(self.csfCb.isChecked())
        )

        """
        aoLb = QLabel('All Output: ')
        allOnPb = QPushButton('ON')
        allOffPb = QPushButton('OFF')
        allOnPb.clicked.connect(lambda: self.setAllOutput(True))
        allOffPb.clicked.connect(lambda: self.setAllOutput(False))
        aoLb.setFixedWidth(60)
        allOnPb.setFixedWidth(25)
        allOffPb.setFixedWidth(28)
        pbSettingLo1.addWidget(aoLb)
        pbSettingLo1.addWidget(allOnPb)
        pbSettingLo1.addWidget(allOffPb)
        pbSettingLo1.addStretch(0)                
        """

        #defaultParam
        w = cmds.getAttr('defaultResolution.width')
        h = cmds.getAttr('defaultResolution.height')
        #if int(cmds.optionVar(q='playblastDisplaySizeSource') ) == 3:
        #    w = cmds.optionVar(q='playblastWidth')
        #    h = cmds.optionVar(q='playblastHeight')
        
        s = self.getOptionVarInit('playblastScale', 0.5)
        pbSettingLo2 = QHBoxLayout()
        sizeLb = QLabel('Image Size: ')
        self.widthLe = QLineEdit(str(w))
        self.heightLe = QLineEdit(str(h))
        scaleLb = QLabel('Playblast Scale: ')
        self.scaleLe = QLineEdit(str(s))
        sizeLb.setFixedWidth(100)
        self.widthLe.setFixedWidth(60)
        self.heightLe.setFixedWidth(60)
        scaleLb.setFixedWidth(89)
        self.scaleLe.setFixedWidth(40)
        pbSettingLo2.addWidget(sizeLb)
        pbSettingLo2.addWidget(self.widthLe)
        pbSettingLo2.addWidget(self.heightLe)
        pbSettingLo2.addSpacing(180)
        pbSettingLo2.addWidget(scaleLb)
        pbSettingLo2.addWidget(self.scaleLe)
        pbSettingLo2.addStretch(0)

        #append feature v1.10
        pbSettingLo4 = QHBoxLayout()
        suffixLb = QLabel('File Name Suffix: ')
        suffixLb.setFixedWidth(100)
        self.suffixLe = QLineEdit()
        self.suffixLe.setFixedWidth(60)
        self.audioCb = QCheckBox()
        self.audioCb.setChecked(0)
        audioLb = QLabel('Use Audio ')
        csfLb.setFixedWidth(90)
        pbSettingLo4.addWidget(suffixLb)
        pbSettingLo4.addWidget(self.suffixLe)
        pbSettingLo4.addSpacing(225)
        pbSettingLo4.addWidget(self.audioCb)
        pbSettingLo4.addWidget(audioLb)
        pbSettingLo4.addStretch(0)    

        #playblast
        pbSettingLo3 = QHBoxLayout()
        loDir = QHBoxLayout()
        lb = QLabel('Save Directory: ')
        lb.setFixedWidth(100)
        self.dirLe = QLineEdit(defaultDir)
        bt = QPushButton('..')
        bt.clicked.connect(self.selectDir)
        loDir.addWidget(lb)
        loDir.addWidget(self.dirLe)
        loDir.addWidget(bt)
        pbSettingLo3.addLayout(loDir)
        pbSettingLo3.addSpacing(30)
        
        self.pbBt = QPushButton('Playblast')
        self.pbBt.setEnabled(False)
        self.pbBt.clicked.connect(self.playblast)
        self.pbBt.setFixedWidth(120)
        self.pbBt.setStyleSheet(
            "QPushButton{background-color : rgba(20,100,80)}"
        )
        pbSettingLo3.addWidget(self.pbBt)

        lo.addLayout(pbSettingLo2)
        lo.addLayout(pbSettingLo1)
        lo.addLayout(pbSettingLo4)
        lo.addLayout(pbSettingLo3)
        
    def offsetPlusMinus(self, pm):
        offset = int(self.offsetLe.text())
        self.offsetLe.setText(str(offset + pm))

    def setAllOutput(self, on):
        if on == -1:
            on = 0
            for rd in self.rowData:
                if rd.outputCb:
                    if not rd.outputCb.isChecked():
                        on = 1
                        break
        for rd in self.rowData:
            if rd.outputCb:
                rd.outputCb.setChecked(on)
                self.setPlayblastButtonEnable()

    def changeItem(self, a):
        if self.tableItemEditable:
            self.rowData[a.row()].changeItem(a.column(), a.text())
    
    def selectItem(self):
        pass            

    def selectDir(self):
        d = cmds.fileDialog2(fm=2, dir=self.dirLe.text())
        self.dirLe.setText(d[0])

    def renameApply(self):
        startNum = self.conv.index('START')
        endNum = self.conv.index('END')
        selRows = set([item.row() for item in self.tbl.selectedItems()])
        for row in selRows:
            start = int(self.tbl.item(row, startNum).text())
            end = int(self.tbl.item(row, endNum).text())
            offset = int(self.offsetLe.text())
            self.tbl.item(row, startNum).setText(str(max(0,start-offset)))
            self.tbl.item(row, endNum).setText(str(max(0,end+offset)))
        self.offsetLe.setText(str(0))

    def playblast(self):         
        msg = QMessageBox()
        msg.setWindowTitle('Confirm')
        msg.setInformativeText('Overwrite it?')
        msg.setIcon(QMessageBox.Question)
        msg.setStandardButtons(QMessageBox.YesToAll | QMessageBox.Yes |
                                 QMessageBox.NoToAll | QMessageBox.No)
        msg.setDefaultButton(QMessageBox.YesToAll)
        toAll = False
        res = 0

        outFiles = []
        cmds.optionVar(stringValue = 
                ('playBlastToolDirName', self.dirLe.text())
        )

        mel.eval("getEditorViewVars()")
        mel.eval("setPlayblastViewVars()")

        isAbort = False
        outputCamNames = [rd.camName for rd in self.rowData if rd.isOutput()]
        outputNum = len(outputCamNames)
        s = '(1/%s) %s'%(outputNum, outputCamNames[0])
        t = 'playblast progress'
        cmds.progressWindow(isInterruptable=1, title=t, status=s, progress=0)
        
        progressFiles = 0
        for rd in self.rowData:
            if cmds.progressWindow(query=1, isCancelled=1) or isAbort:
                isAbort = True
                break
        
            if rd.isOutput():
                #progress
                p = 100.0*progressFiles/outputNum
                progressFiles += 1
                s = '(%s/%s) %s'%(progressFiles, outputNum, rd.camName)
                cmds.progressWindow(e=1, status=s, progress=p)
                #cmds.pause(seconds=0.1)

                isOutput = True
                rd.setkw()
                self.vHeaderClicked(rd.row, sel=False)

                isImageSameName = False
                if rd.kw['format'] == 'image':
                    for i in range(rd.kw['startTime'], rd.kw['endTime']+1):
                        imageName ='%s.%s%s'%(
                                rd.kw['filename'], 
                                str(i).zfill(rd.kw['framePadding']),
                                rd.extension)
                        if os.path.isfile(imageName):    
                            isImageSameName = True
                            break
                if  (os.path.isfile(rd.kw['filename']) and 
                     rd.kw['format'] != 'image') or isImageSameName:
                    if not toAll:
                        if rd.kw['format'] != 'image':
                            msg.setText('%s already exists.'%rd.movieFileName)
                        else:
                            msg.setText('%s.%s%s already exists.'%(
                                    rd.movieFileName,
                                    '#' * rd.kw['framePadding'],
                                    rd.extension)
                                )
                        res = msg.exec_()
                    if res in [QMessageBox.YesToAll, QMessageBox.NoToAll]:
                        toAll = True
                    if res in [QMessageBox.No, QMessageBox.NoToAll]:
                        isOutput = False
                if isOutput:
                    #calc start frame
                    if self.csfCb.isChecked():
                        _s = int(math.floor(float(self.csfLe.text())))
                        for i in range(_s, rd.start):
                            cmds.currentTime(i)
                            if cmds.progressWindow(query=1, isCancelled=1):
                                isAbort = True
                                break

                    #playblast
                    #pre
                    ovs = '%s.overscan'%rd.camName
                    preOsLock = cmds.getAttr(ovs, l=1)
                    preConnectAttr = cmds.listConnections(ovs, s=1, d=0, p=1)
                    if preConnectAttr:
                        cmds.disconnectAttr(preConnectAttr[0], ovs)
                    cmds.setAttr(ovs, l=0)
                    preOs = cmds.getAttr(ovs)  
                    if 'OVERSCAN' in self.conv: 
                        cmds.setAttr(ovs, rd.overscan/100.0)

                    #exec
                    cmds.currentTime(rd.kw['startTime'])
                    cmds.playblast(**rd.kw)
                    #print ('PLAYBLAST:',rd.kw)
                    if rd.kw['format'] == 'image':
                        outFiles.append('%s.[%s-%s]%s'%(
                            rd.movieFileName, 
                            rd.kw['startTime'], 
                            rd.kw['endTime'], 
                            rd.extension)
                        )
                    else:
                        outFiles.append(rd.movieFileName)
                    
                    #post
                    cmds.setAttr(ovs, preOs)
                    cmds.setAttr(ovs, l=preOsLock)
                    if preConnectAttr:
                        cmds.connectAttr(preConnectAttr[0], ovs)
                    #set optionVar
                    #cmds.optionVar(fv=("playblastScale", float(rd.kw['percent'])/100.0))

        mel.eval("restoreEditorViewVars()")
        cmds.progressWindow(endProgress=1)
        if outFiles:
            if isAbort:
                QMessageBox.information(self, 'Canceled', 
                 'playblasted following files:\n\n'+'\n'.join(outFiles[:-1])
                 +'\n\n\n'+
                 'canceled following file:\n\n'+outFiles[-1]
                )
            else:
                QMessageBox.information(self, 'Completed', 
                'playblasted following files:\n\n'+'\n'.join(outFiles))

    def vHeaderClicked(self, num, sel=True):
        rd = self.rowData[num]
        cmds.lookThru(rd.camName)
        if sel:
            cmds.select(rd.camName, r=1)
        
        if rd.noError:
            rd.setkw()
            cmds.currentUnit(time=rd.fps)
            
            #with offset
            cmds.playbackOptions(minTime = rd.kw['startTime'])
            cmds.playbackOptions(maxTime = rd.kw['endTime'])
            #without offset
            #cmds.playbackOptions(minTime = rd.start)
            #cmds.playbackOptions(maxTime = rd.end)

    def getTableItem(self, row, col):
        if type(col) == int:
            return self.tbl.item(row,col).text()
        else:
            if col in self.conv :
                return self.tbl.item(row, self.conv.index(col)).text()
        return None
                
    def getParentVis(self, n):
        n = [n]
        safety = 0
        while cmds.listRelatives(n[-1], p=1) and safety < 20:
            safety += 1
            n.append(cmds.listRelatives(n[-1],p=1,f=1)[0])
        
        for _n in n:
            if not cmds.getAttr('%s.visibility'%_n):
                return False
        return True
    class tableRowItems:
        def __init__(self, gui, cam, row):
            self.gui = gui
            self.tbl = self.gui.tbl
            self.conv = self.gui.conv
            self.row = row
            self.camName = cam
            self.reset()
        
        def changeItem(self, col, text):
            newName = []
            for i in range(len(self.conv)):
                if i != col:
                    newName.append(self.tbl.item(self.row, i).text())
                else:
                    newName.append(text)
            
            newName = '_'.join(newName)
            cmds.rename(self.camName, newName)
            self.camName = newName
            
            self.resetWithCheckState()

            self.tbl.setVerticalHeaderLabels(
                [rd.camName for rd in self.gui.rowData]
            )
        
        def resetWithCheckState(self):
            newCbState = True
            if self.outputCb:
                newCbState = self.outputCb.checkState()
            self.reset()
            if self.outputCb:
                self.outputCb.setChecked(newCbState)


        def reset(self):
            self.scene = None
            self.cut = None
            self.start = None
            self.end = None
            self.width = None
            self.height = None
            self.fps = None
            self.overscan = None
            self.outputCb = None
            self.matchNum = False
            self.outputCbWid = None
            self.outputCbLo = None
            self.kw = {}

            self.noError = True
            self.matchNum = (len(self.conv) == len(self.camName.split('_')))
            self.noError *= self.matchNum

            self.gui.tableItemEditable = False
            #set item
            for j in range(len(self.conv)):
                item = QTableWidgetItem()
                item.setTextAlignment(Qt.AlignCenter)
                if self.matchNum:
                    itemName = self.camName.split('_')[j]
                    item.setText(itemName)

                    #rowData error
                    c = self.conv[j]
                    intSet = ('START','END','WIDTH','HEIGHT','OVERSCAN')
                    tmp = not(c in intSet and not isInt(itemName))
                    tmp *= not(c == 'FPS' and not isFPS(itemName))
                    paramSet = ('START','END','WIDTH','HEIGHT','OVERSCAN','FPS')
                    tmp *= not(c in paramSet and self.conv.count(c) > 1)

                    if tmp:
                        item.setBackground(QColor(*self.gui.RGB_OK))
                    else:
                        item.setBackground(QColor(*self.gui.RGB_NG))
                        self.noError = False
                    
                else:
                    item.setBackground(QColor(*self.gui.RGB_NG))
                    item.setFlags(Qt.ItemIsEnabled)
                self.tbl.setItem(self.row, j, item)

            #del previous checkBox
            for j in range(len(self.conv)+1):
                self.tbl.removeCellWidget(self.row, j)

            #set checkBox
            if self.noError:
                self.outputCb = QCheckBox(parent = self.gui)

                self.outputCb.setChecked(self.gui.getParentVis(self.camName))
                self.outputCbWid = QWidget()
                self.outputCbLo = QHBoxLayout(self.outputCbWid)
                self.outputCbLo.addWidget(self.outputCb)
                self.outputCbLo.setAlignment(Qt.AlignCenter)
                self.outputCb.setStyleSheet('QCheckBox{background-color: #222222;}')
                #self.outputCbWid.setStyleSheet('QWidget{background-color: rgb%s;}'%(str(self.gui.RGB_OK)) )
                self.tbl.setItem(self.row, len(self.conv), QTableWidgetItem())
                self.outputCb.clicked.connect(self.gui.setPlayblastButtonEnable)

                self.tbl.setCellWidget(self.row, len(self.conv), self.outputCbWid)
                self.tbl.item(self.row, len(self.conv)).setBackground(QColor(*self.gui.RGB_OK))
            else:
                item = QTableWidgetItem()
                item.setTextAlignment(Qt.AlignCenter)
                item.setBackground(QColor(*self.gui.RGB_NG))
                item.setFlags(Qt.ItemIsEnabled)
                self.tbl.setItem(self.row, len(self.conv), item)

            
            self.gui.tableItemEditable = True

        def isOutput(self):
            if self.outputCb:
                return self.outputCb.isChecked()
            else:
                return False
        def existsAudio(self):
            return bool(cmds.ls(type='audio'))
            
        def setkw(self):
            if self.noError:
                sp = self.camName.split('_')
                for i,r in enumerate(self.conv):
                    if r == 'START'   : self.start = int(sp[i])
                    if r == 'END'     : self.end = int(sp[i])
                    if r == 'WIDTH'   : self.width = int(sp[i])
                    if r == 'HEIGHT'  : self.height = int(sp[i])
                    if r == 'OVERSCAN': self.overscan = float(sp[i])
                    if r == 'FPS'     : self.fps = sp[i]

            # default        
            #self.kw['startTime'] = int(cmds.optionVar(q="playblastStartTime"))
            #self.kw['endTime'] = int(cmds.optionVar(q="playblastEndTime"))
            self.kw['startTime'] = int(cmds.getAttr("defaultRenderGlobals.startFrame"))
            self.kw['endTime'] = int(cmds.getAttr("defaultRenderGlobals.endFrame"))
            self.kw['width'] = int(self.gui.widthLe.text())
            self.kw['height'] = int(self.gui.heightLe.text())
            self.kw['percent'] = float(self.gui.scaleLe.text())*100.0
            if self.existsAudio():
                self.kw['useTraxSounds'] = self.gui.audioCb.isChecked()

            if 'START'    in self.conv: self.kw['startTime'] = self.start
            if 'END'      in self.conv: self.kw['endTime'] = self.end
            if 'WIDTH'    in self.conv: self.kw['width'] = self.width
            if 'HEIGHT'   in self.conv: self.kw['height'] = self.height

            self.kw['compression'] = cmds.optionVar(q="playblastCompression")
            self.kw['format'] = cmds.optionVar(q="playblastFormat")
            self.extension = {'avi':'.avi', 'qt':'.mov', 
                'image':imageExt(self.kw['compression'])}[self.kw['format']]
            sceneName = cmds.file(sn=1,shn=1,q=1).split('.')[:-1]
            self.dirName = self.gui.dirLe.text()
            if not self.dirName:
                ws = cmds.workspace(q=1,active=1)
                u = os.environ.get('USERNAME')
                mvdir = 'C:/Users/%s/Documents/maya/projects/default/movies'%u
                if os.path.exists(ws):
                    self.dirName = ws+'/movies'
                elif os.path.exists(mvdir):
                    self.dirName = mvdir

            if self.kw['format'] == 'image':
                self.movieFileName = '%s%s'%(
                   self.camName, self.gui.suffixLe.text()
                )
            else:
                self.movieFileName = '%s%s%s'%(
                    self.camName, self.gui.suffixLe.text(), self.extension
                )
            if sceneName:
                self.movieFileName = '%s_%s'%(
                    ".".join(sceneName), self.movieFileName
                )
            self.kw['forceOverwrite'] = True
            self.kw['filename'] = '%s/%s'%(self.dirName, self.movieFileName)
            
            self.kw['framePadding'] = self.gui.getOptionVarInit("playblastPadding", 4)
            self.kw['quality'] = self.gui.getOptionVarInit("playblastQuality", 70)
            self.kw['offScreen'] = self.gui.getOptionVarInit("playblastOffscreen", 0)
            self.kw['viewer'] = 0
            self.kw['showOrnaments'] = self.gui.getOptionVarInit("playblastShowOrnaments", 1)
            # Offset time range
            self.kw['startTime'] -= int(self.gui.offsetLe.text())
            self.kw['endTime'] += int(self.gui.offsetLe.text())
    
    def getOptionVarInit(self, opt, init):
        ret = init
        if cmds.optionVar(ex=opt):
            ret = cmds.optionVar(q=opt)
        return ret

    def updateConv(self, camNames=None):
        if camNames is None:
            camNames = self.getSelCams()
        self.tbl.setRowCount(len(camNames))
        self.tbl.setVerticalHeaderLabels(camNames)
        self.tbl.setColumnCount(len(self.conv)+1)
        self.tbl.setHorizontalHeaderLabels(self.conv+['Output'])
        self.tbl.verticalHeader().sectionClicked.connect(self.vHeaderClicked)
        

        #set columnWidth
        for j in range(len(self.conv)):
            if self.conv[j] == '*':
                self.tbl.setColumnWidth(j, 0)
            elif len(self.conv[j]) > 6 :
                w = (len(self.conv[j]) - 6) * 8 + self.COLUMN_WIDTH_ITEM
                self.tbl.setColumnWidth(j, w)
            else:
                self.tbl.setColumnWidth(j, self.COLUMN_WIDTH_ITEM)
        self.tbl.setColumnWidth(j+1, self.COLUMN_WIDTH_OUTPUT)

        #set rowData
        self.rowData = [] 
        for i,camName in enumerate(camNames):
            self.rowData.append(self.tableRowItems(self, camName, i))
        

        #widen window height
        h = self.size().height()
        w = self.size().width()        
        new_h = 330 + len(camNames)*30
        if h < new_h:
            h = new_h
            self.resize(w,h)
            if h > self.WINDOW_HEIGHT_MAX:
                self.resize(w,self.WINDOW_HEIGHT_MAX)
        
        #update UI
        self.appPb.setEnabled('START' in self.conv and 'END' in self.conv)
        self.widthLe.setEnabled(not 'WIDTH' in self.conv)
        self.heightLe.setEnabled(not 'HEIGHT' in self.conv)
        self.setPlayblastButtonEnable()

    def setPlayblastButtonEnable(self):
        self.pbBt.setEnabled(False)
        for rd in self.rowData:
            if rd.outputCb:
                if rd.outputCb.checkState():
                    self.pbBt.setEnabled(True)
    
    def selectNowCams(self):
        cmds.select(self.getNowCams(), r=1)

    def getNowCams(self):
        return self.retCamNode([rd.camName for rd in self.rowData])

    def setCams(self):
        self.updateConv(self.getSelCams())

    def addCams(self):
        cams = self.getNowCams() + self.getSelCams()
        self.updateConv(sorted(set(cams), key=cams.index))

    def remCams(self):
        cams = [cam for cam in self.getNowCams() 
                if not cam in self.getSelCams(children=False)
                if not cam in 
                    [self.rowData[i].camName for i in 
                        [item.row() for item in self.tbl.selectedItems()]
                    ]
                ]
        self.updateConv(cams)

    def getSelCams(self, children=True):
        if not cmds.selectPref(tso=True, q=1):
            cmds.selectPref(tso=True)
        return self.retCamNode(cmds.ls(os=1, dag=children, type='transform'))
        
    def retCamNode(self,nodes):
        nodes = [node for node in nodes if cmds.objExists(node)]
        ts= [t for t in nodes
                if cmds.listRelatives(t, shapes=1)
            ]
        return [t for t in ts if 'camera' in 
                    [cmds.nodeType(s) for s in cmds.listRelatives(t, shapes=1, f=1)]
                    if self.getParentVis(t)
                ]
                
def isInt(s):
    try:
        int(s)
    except ValueError as _:
        return False
    else:
        return True
def isFPS(s):
    try:
        cmds.currentUnit(time=s)
    except RuntimeError as _:
        return False
    else:
        return True
def imageExt(s):
    extDir = {
        'global':'',
        'si':'.pic',
        'tifu':'.tif',
        'maya':'.iff',
        'psdLayered':'.psd'}
    if s in list(extDir):
        return extDir[s]
    return '.'+str(s)

def createWindow():
    ui = GUI()
    wid = QApplication.topLevelWidgets()
    for i in wid:
        if ui.objectName() == i.objectName():
            i.close()

    ui.setAttribute(Qt.WA_DeleteOnClose)
    ui.show()
    return ui
    
if __name__ == '__main__':
    createWindow()
