# coding=utf-8
#
# Copyright (c) 2023 LIBZENT Innovations, Inc. BACKBONEstudio.
# Copyright (c) 2023 LIBZENT BIGFOOT Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
#
# Author:
#   LIBZENT Innovations, Inc. BACKBONEstudio
#   BIGFOOT Inc.
# date:
#   2024/01/23 v1.00 release

import bpy
import blf
import math
import sys
import os
import json
import inspect
import addon_utils
import subprocess
import site
import importlib
import re

from mathutils import *
from bpy.types import Panel, Operator
from bpy.app.handlers import persistent
from bpy_extras.io_utils import ImportHelper, ExportHelper
from bpy.props import StringProperty, BoolProperty, EnumProperty, FloatProperty,IntProperty

try:
    print("-----Normal Process Start-----")
    #out = subprocess.run([sys.executable, '-m', 'pip', 'show', 'PySide2'], check=True,
    #        capture_output=True, text=True)
    #sitePackagesPath = re.search(r"^Location: (.+)$", out.stdout, re.MULTILINE).group(1)
    script_file = os.path.realpath(__file__)
    directory = os.path.dirname(script_file)
    sitePackagesPath = os.path.join(directory, 'modules', 'site-packages')
    print('--sitePackage path:', sitePackagesPath)
    if not sitePackagesPath in sys.path:
        sys.path.insert(0,sitePackagesPath)
    print(sys.path)
    import PySide2
    print(PySide2)
    plugin_path = os.path.join(sitePackagesPath,'PySide2', 'plugins', 'platforms')
    os.environ['QT_QPA_PLATFORM_PLUGIN_PATH'] = plugin_path
    from PySide2.QtCore import Qt, QEventLoop
    from PySide2 import QtWidgets
    print("Successfully imported PySide2.")
    print("Added Path:", plugin_path)
    print("-----Normal Process End-----")
except:
    print("-----Import Error Process Start-----")
    print("Run pip upgrade.")
    #subprocess.run([sys.executable, '-m', 'pip', 'install', '--upgrade', 'pip'], check=True)
    #print("----install Pyside2----")
    #subprocess.run([sys.executable, '-m', 'pip', 'install', '--user', 'PySide2'], check=True)
    #out = subprocess.run([sys.executable, '-m', 'pip', 'show', 'PySide2'], check=True,
    #        capture_output=True, text=True)
    script_file = os.path.realpath(__file__)
    directory = os.path.dirname(script_file)
    sitePackagesPath = os.path.join(directory, 'modules', 'site-packages')
    sitePackagesPath = re.search(r"^Location: (.+)$", out.stdout, re.MULTILINE).group(1)
    print('--sitePackage path:', sitePackagesPath)
    if not sitePackagesPath in sys.path: 
        sys.path.insert(0,sitePackagesPath)
    print(sys.path)    
    import PySide2
    print("----import PySide Module Clear----")
    plugin_path = os.path.join(sitePackagesPath,'PySide2', 'plugins', 'platforms')
    os.environ['QT_QPA_PLATFORM_PLUGIN_PATH'] = plugin_path
    from PySide2.QtCore import Qt, QEventLoop
    from PySide2 import QtWidgets
    print("Successfully installed PySide2.")
    print("-----Import Error Process End-----")

try:
    import datetime
    print(datetime)
    print("Successfully imported datetime.")
except ImportError:
    print("Run pip upgrade.")
    subprocess.run([sys.executable, '-m', 'pip', 'install', '--upgrade', 'pip'], check=True)
    subprocess.run([sys.executable, '-m', 'pip', 'install', '--user', 'datetime'], check=True)
    import datetime

import imp
try:
    imp.find_module('PySide2')
    import PySide2
    from PySide2.QtWidgets import *
    from PySide2.QtGui import *
    from PySide2.QtCore import *
    from PySide2.QtUiTools import *
    import shiboken2
    from shiboken2 import wrapInstance
except ImportError:
    import PySide
    from PySide.QtGui import *
    from PySide.QtCore import *
    from PySide.QtUiTools import *
    import shiboken
    from shiboken import wrapInstance

import copy


bl_info = {
    "name": "HUD custom tools for Blender",
    "author": "LIBZENT Innovations, Inc. BACKBONEstudio / BIGFOOT Inc.",
    "version": (1,0,0),
    "blender": (3, 6, 5),
    "warning": "",
    "support": "COMMUNITY",
    "category": "View3D"
}


path = os.path.join(os.path.dirname(__file__))
path = os.path.join(path,'HUDCustomToolsGUIForBlender')



path = path.replace('\\','/')
if not dir in sys.path:
    sys.path.append(path)
import settings as s
imp.reload (s)

camSpeedUnit = 'mph'
velSpeedUnit = 'mph'
distanceUnit = 'm '
velNodes = ['']*5
distanceNodes = ['']*5
objVelocities = [('',0)] * 6
preObjPos =[Vector((0,0,0))]*6
tmpObjPos = [Vector((0,0,0))]*6
currentObjPos = [Vector((0,0,0))]*6
pre_frame = 0

def regist_props():
    scene = bpy.types.Scene
    scene.hud_xOffset = IntProperty(
        name="hud_xOffset",
        description="hud_xOffset",
        default=4,
        min=0,
        max=100
    )
    scene.hud_yOffset = IntProperty(
        name="hud_yOffset",
        description="hud_yOffset",
        default=4,
        min=0,
        max=100
    )
    scene.enable_Note_on_Save = BoolProperty(
        name="enable_Note_on_Save",
        description="Note hidden on save",
        default=False,
        update=update_Save_handler
    )

def unregist_props():
    scene = bpy.types.Scene
    del scene.hud_xOffset
    del scene.hud_yOffset
    del scene.enable_Note_on_Save

@persistent
def enable_Note_on_Save_pre_handler(dammy):
    if bpy.context.scene.enable_Note_on_Save == False:
        bpy.context.scene.render.use_stamp_note = False
    else:
        pass

@persistent
def enable_Note_on_Save_post_handler(dammy):
    if bpy.context.scene.enable_Note_on_Save == False:
        bpy.context.scene.render.use_stamp_note = True
    else:
        pass


def update_Save_handler(self, context):
    if self.enable_Note_on_Save == False:
        if enable_Note_on_Save_pre_handler not in bpy.app.handlers.save_pre:
            bpy.app.handlers.save_pre.append(enable_Note_on_Save_pre_handler)
        if enable_Note_on_Save_post_handler not in bpy.app.handlers.save_post:
            bpy.app.handlers.save_post.append(enable_Note_on_Save_post_handler)
    else:
        if enable_Note_on_Save_pre_handler in bpy.app.handlers.save_pre:
            bpy.app.handlers.save_pre.remove(enable_Note_on_Save_pre_handler)
        if enable_Note_on_Save_post_handler in bpy.app.handlers.save_post:
            bpy.app.handlers.save_post.remove(enable_Note_on_Save_post_handler)


def ShowMessageBox(message = "", title = "Message Box", icon = 'INFO'):

    def draw(self, context):
        self.layout.label(text=message)
    bpy.context.window_manager.popup_menu(draw, title = title, icon = icon)

class RENDER_TEXT():
    blfHUD_show = False
    hudSB = []
    hudText = []
    fontSize = 16
    def get_note_length():
        note_length = len(bpy.context.scene.render.stamp_note_text)
        return note_length

    def get_font_info(self,GUI):
        self.font_info = {
            "font_id": 0,
            #color
            "col_r":bpy.context.scene.render.stamp_foreground[0],
            "col_g":bpy.context.scene.render.stamp_foreground[1],
            "col_b":bpy.context.scene.render.stamp_foreground[2],
            "col_a":bpy.context.scene.render.stamp_foreground[3],
            #size
            "size":self.fontSize,
            #pos
            "pos_x_offset":bpy.context.scene.hud_xOffset,
            "pos_y_offset":bpy.context.scene.hud_yOffset,
            "pos_z_offset":0,
            #offset
            "line_height":self.fontSize,
            "x_offset":0,
            "y_offset":0,
        }

        return self.font_info  

    @persistent
    def draw_view3d_callback_px(self,area_w,area_h,font_info,GUI):
        font_id = font_info["font_id"]
        x_basepos = round(area_w*(bpy.context.scene.hud_xOffset * 0.01))
        y_basepos = round((area_h*(1-((bpy.context.scene.hud_yOffset) * 0.01)))\
                          -font_info["size"]-30)
        z_basepos = round(font_info["pos_z_offset"])
        blf.position(font_id,x_basepos,y_basepos,z_basepos)
        try:
            blf.size(font_id, font_info["size"])
        except:
            blf.size(font_id, font_info["size"],72)
        blf.color(font_id,font_info["col_r"],font_info["col_g"],\
                  font_info["col_b"],font_info["col_a"])

        packed_text = VariousCalculation.collect_HUD_text(GUI)
        try:
            columns = {}
            stamp_text = ""
            for item in packed_text:
                name, _, col = item
                if col not in columns:
                    columns[col] = []
                columns[col].append(name)

            for col, names in columns.items():
                names_str = '    '.join(names)
                self.y_offset =  -(self.font_info["line_height"] * col)
                y_pos = y_basepos + self.y_offset
                blf.position(font_id,x_basepos,y_pos,z_basepos)
                try:
                    blf.draw(font_id,str(names_str))
                except:
                    blf.draw(font_id,str("draw Eroor"))

                
        except:
            packed_text = "HUD draw Error"
            blf.draw(font_id,str(packed_text))

    def redraw_regions():
        for area in bpy.context.window.screen.areas:
            if area.type == 'VIEW_3D':
                for region in area.regions:
                    if region.type == 'WINDOW':
                        region.tag_redraw()


    def stamp_entry(scene):
        packed_text = VariousCalculation.collect_HUD_text(HUD_View3d.ui)
        if RENDER_TEXT.blfHUD_show == True:
            try:
                columns = {}
                previous_columns = 0
                stamp_text = ""
                for item in packed_text:
                    name, _, col = item
                    if col not in columns:
                        columns[col] = []
                    columns[col].append(name)
                
                sorted_columns = dict(sorted(columns.items(), key=lambda x: x[0]))
                
                for col, names in sorted_columns.items():
                    names_str = '    '.join(names)
                    #
                    newlines = col - previous_columns
                    previous_columns = col
                    nl = ''
                    if newlines < 1:
                        newlines = 0
                    else:
                        pass
                    for n in range(newlines):
                        nl = nl + '\n'

                    try:
                        stamp_text = stamp_text + nl + str(names_str)
                    except:
                        stamp_text = stamp_text + nl + str("draw Eroor")

                bpy.context.scene.render.stamp_note_text = str(stamp_text)
            except:
                packed_text = "HUD draw Error"
                bpy.context.scene.render.stamp_note_text = str(packed_text)
        else:
            bpy.context.scene.render.stamp_note_text = ""                        
                        
    
    def delete_driver_handler(handlername):
        blf_handler = bpy.app.driver_namespace.get(handlername)
        if blf_handler:
            bpy.types.SpaceView3D.draw_handler_remove(blf_handler,'WINDOW')
            del bpy.app.driver_namespace[handlername]
    
    def add_driver_handler(self,area_w,area_h,handlername,font_info,GUI):
        blf_handler = bpy.app.driver_namespace.get(handlername)
        if not blf_handler:
            blf_handler = bpy.types.SpaceView3D.draw_handler_add(
                self.draw_view3d_callback_px,(RENDER_TEXT,area_w,area_h,font_info,GUI),'WINDOW','POST_PIXEL')
            bpy.app.driver_namespace[handlername] = blf_handler


    @persistent
    def load_pre_handler(handlername):
        blf_handler = bpy.app.driver_namespace.get("draw_HUD_View3d")
        if blf_handler:
            bpy.types.SpaceView3D.draw_handler_remove(blf_handler,'WINDOW')
            del bpy.app.driver_namespace["draw_HUD_View3d"]
        try:
            HUD_View3d.ui.close()
            HUD_View3d.ui.deleteLater()
        except:
            pass

    def add_load_pre_handler(handlername):
        blf_handler = bpy.app.driver_namespace.get(handlername)
        if not blf_handler:
            bpy.app.handlers.load_pre.append(RENDER_TEXT.load_pre_handler)

    def del_load_pre_handler(handlername):
        blf_handler = bpy.app.driver_namespace.get(handlername)
        if not blf_handler:
            for handler in bpy.app.handlers.load_pre:
                if handler.__name__ == "draw_HUD_View3d":
                    bpy.app.handlers.load_pre.remove(handler)
                else:
                    pass
        try:
            del bpy.app.driver_namespace[handlername]
        except:
            pass


class HUDTOOL_PT_Panel(bpy.types.Panel):
    bl_label = "HUD custom tools"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"
    bl_category = 'HUD'

    def draw(self, context):
        self.layout.operator('hudtools.show_gui')       
        self.layout.operator("hud_toggle.button")
        self.layout.operator('hud_setup.button')
        self.layout.separator()
        self.layout.prop(context.scene, "enable_Note_on_Save", text="enable Note on save")

        self.layout.separator()
        self.layout.label(text="HUD Offset")
        row = self.layout.row()
        row.alignment = 'EXPAND'
        row.prop(context.scene, "hud_xOffset", text="X:")
        row.prop(context.scene, "hud_yOffset", text="Y:")
        self.layout.separator()
        self.layout.operator('hudtools.import_jsonbutton')
        self.layout.operator('hudtools.export_jsonbutton')
        self.layout.separator()


class VariousCalculation():
    pre_frame = 0
    VLnote_length = 0
    def get_addon_path(self):
        for mod in addon_utils.modules():
            if mod.bl_info['name'] == "CustomHUDTool":
                filepath = mod.__file__
                return filepath
            else:
                pass
        return None
    
    def get_area_size():
        for region in bpy.context.area.regions:
            if region.type == 'WINDOW':
                area_w = region.width
                area_h = region.height
        return area_w,area_h

    def update_HUD_SB(GUI):
        RENDER_TEXT.hudSB = []
        for b in GUI.buttons:
            if b.setting.vis:
                nam = b.setting.nam
                sec = b.setting.section
                blc = b.setting.block
                RENDER_TEXT.hudSB.append((b,nam,sec,blc))
        
        return RENDER_TEXT.hudSB

    def collect_HUD_text(GUI):
        RENDER_TEXT.hudText = []
        for SB in RENDER_TEXT.hudSB:
            for b in GUI.buttons:
                if SB[1] == b.setting.nam:
                    if b.setting.cmd == '':
                        text = b.setting.lab
                        RENDER_TEXT.hudText.append((text,SB[2],SB[3]))
                    else:
                        if b.setting.evt == '':
                            try:
                                text = b.setting.lab+str((eval(b.setting.cmd)))
                            except:
                                text = b.setting.lab+str((eval(b.setting.cmd)))
                                pass
                            RENDER_TEXT.hudText.append((text,SB[2],SB[3]))
                        else:
                            try:
                                text = b.setting.lab+str((eval(b.setting.cmd)))
                            except:
                                text = b.setting.lab+str((eval(b.setting.cmd)))
                                pass
                            RENDER_TEXT.hudText.append((text,SB[2],SB[3]))
                    break
        return RENDER_TEXT.hudText
    
    def get_note_length(GUI):
        collected_button_text = []
        VariousCalculation.VLnote_length = 0
        for b in GUI.buttons:
            if b.setting.vis == True:
                if b.setting.cmd == '':
                    text = b.setting.lab
                    collected_button_text.append((text,b.setting.section,b.setting.block))
                else:
                    if b.setting.evt == '':
                        try:
                            text = b.setting.lab+str((eval(b.setting.cmd)))
                            collected_button_text.append((text,b.setting.section,b.setting.block))
                        except:
                            text = b.setting.lab+str((eval(b.setting.cmd)))
                            collected_button_text.append((text,b.setting.section,b.setting.block))
                            pass
                    else:
                        try:
                            text = b.setting.lab+str((eval(b.setting.cmd)))
                            collected_button_text.append((text,b.setting.section,b.setting.block))
                        except:
                            text = b.setting.lab+str((eval(b.setting.cmd)))
                            collected_button_text.append((text,b.setting.section,b.setting.block))
                            pass

        columns = {}
        previous_columns = 0
        collected_note_text = ""
        for item in collected_button_text:
            name, _, col = item
            if col not in columns:
                columns[col] = []
            columns[col].append(name)
        
        sorted_columns = dict(sorted(columns.items(), key=lambda x: x[0]))
        
        for col, names in sorted_columns.items():
            names_str = '    '.join(names)
            newlines = col - previous_columns
            previous_columns = col
            nl = ""
            if newlines < 1:
                newlines = 0
            else:
                pass
            for n in range(newlines):
                nl = nl + "\n"

            
            collected_note_text = collected_note_text + nl + str(names_str)
        return len(collected_note_text.encode('utf-8'))
    
    def check_exists_object(self,objectname):
        for obj in bpy.context.scene.objects:
            if objectname == obj.name_full:
                return True
        return False

    def get_systemunit(self):
        if bpy.context.scene.unit_settings.system == 'NONE':
            systemunit = ["",1.0]
            return systemunit
        
        elif bpy.context.scene.unit_settings.system == 'METRIC':
            if bpy.context.scene.unit_settings.length_unit == 'ADAPTIVE':
                systemunit = ["m",1.0]
            elif bpy.context.scene.unit_settings.length_unit == 'KILOMETERS':
                systemunit = ["km",0.001]
            elif bpy.context.scene.unit_settings.length_unit == 'METERS':
                systemunit = ["m",1.0]
            elif bpy.context.scene.unit_settings.length_unit == 'CENTIMETERS':
                systemunit = ["cm",100.0]
            elif bpy.context.scene.unit_settings.length_unit == 'MILLIMETERS':
                systemunit = ["mm",1000.0]
            elif bpy.context.scene.unit_settings.length_unit == 'MICROMETERS':
                systemunit = ["um",1000000.0]

            return systemunit
        
        elif bpy.context.scene.unit_settings.system == 'IMPERIAL':
            if bpy.context.scene.unit_settings.length_unit == 'ADAPTIVE':
                systemunit = ["ft",3.28084]
            elif bpy.context.scene.unit_settings.length_unit == 'MILES':
                systemunit = ["mi",0.000621]
            elif bpy.context.scene.unit_settings.length_unit == 'FEET':
                systemunit = ["ft",3.28084]
            elif bpy.context.scene.unit_settings.length_unit == 'INCHES':
                systemunit = ["in",39.3701]
            elif bpy.context.scene.unit_settings.length_unit == 'THOU':
                systemunit = ["mil",39370.1]
            return systemunit
        

    def get_distance(vec1,vec2):
        dist = (vec2 - vec1).length
        return dist
    
    def get_filename(filepath):
       
        if bpy.path.basename(filepath) == '':
            return str("")
        else:
             return str((bpy.path.basename(filepath)).split('.')[:-1][0])
    
    def get_scene_camera():
        if bpy.context.scene.camera == None:
            return '---------'
        else:
            camera_name = bpy.context.scene.camera.name_full
            return camera_name

    def get_camera_lens():
        if bpy.context.scene.camera == None:
            return '---------'
        else:
            camera_lens = '{:>5.3f} mm'.format(bpy.context.scene.camera.data.lens)
            return camera_lens
    
    def get_camera_height(self):
        if bpy.context.scene.camera == None:
            return '---------'
        else:
            height_world = bpy.context.scene.camera.matrix_world.translation[2]
            sysunit = self.get_systemunit(self)
            height_world = height_world * sysunit[1]
            camera_height = '{:>9.3f} {}'.format(float(height_world), sysunit[0])
            return camera_height
        
    def get_camera_location(self,axis):
        if bpy.context.scene.camera == None:
            return '---------'
        else:
            sysunit = self.get_systemunit(self)
            if axis == 0:
                cam_world_location = bpy.context.scene.camera.matrix_world.translation[0]
            elif axis == 1:
                cam_world_location = bpy.context.scene.camera.matrix_world.translation[1]
            elif axis == 2:
                cam_world_location = bpy.context.scene.camera.matrix_world.translation[2]
            cam_world_location = cam_world_location * sysunit[1]
            camera_location = '{:>9.3f} {}'.format(float(cam_world_location), sysunit[0])
            return camera_location
    
    def get_camera_rotation(self,axis,mode):
        if bpy.context.scene.camera == None:
            return '---------'
        elif mode == 0:
            if axis == 0:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.matrix_world.to_quaternion().to_euler()[0])
            elif axis == 1:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.matrix_world.to_quaternion().to_euler()[1])
            elif axis == 2:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.matrix_world.to_quaternion().to_euler()[2])

        elif mode == 1:
            if axis == 0:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.rotation_euler[0])
            elif axis == 1:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.rotation_euler[1])
            elif axis == 2:
                camera_base_rotation = math.degrees(bpy.context.scene.camera.rotation_euler[2])

        camera_rotation = '{:7.1f}°'.format(float(camera_base_rotation))
        return camera_rotation


    def get_frame_curret():
        currentframe = str(bpy.context.scene.frame_current)
        return currentframe
    
    def get_previous_position(self):
        for i in range(6):
            elapsed_time = 0
            if i == 0:
                if bpy.context.scene.camera:
                    if VariousCalculation().check_exists_object(bpy.context.scene.camera.name_full) == True:
                        if VariousCalculation.pre_frame == bpy.context.scene.frame_current:
                            pass
                        else:
                            preObjPos[0] = tmpObjPos[0]
                            currentObjPos[0] = bpy.context.scene.camera.matrix_world.translation
                            tmpObjPos[0] = copy.deepcopy(currentObjPos[0])
                            objVelocities[0] = (bpy.context.scene.camera.name_full, (currentObjPos[0]-preObjPos[0]).length / (bpy.context.scene.frame_current - VariousCalculation.pre_frame))
                    else:
                        objVelocities[0] = ("",0)
                else:
                    objVelocities[0] = ("",0)
            else:
                if VariousCalculation().check_exists_object(velNodes[i-1]) == True:
                    if VariousCalculation.pre_frame == bpy.context.scene.frame_current:
                        pass
                    else:
                        for sceneobj in bpy.context.scene.objects:
                            if velNodes[i-1] == sceneobj.name_full:
                                obj = sceneobj
                        preObjPos[i] = tmpObjPos[i]
                        currentObjPos[i] = obj.matrix_world.translation
                        tmpObjPos[i] = copy.deepcopy(currentObjPos[i])
                        objVelocities[i] = (velNodes[i-1], (currentObjPos[i]-preObjPos[i]).length / (bpy.context.scene.frame_current - VariousCalculation.pre_frame))
                else:
                    objVelocities[i] = ("",0)
        VariousCalculation.pre_frame = bpy.context.scene.frame_current
        return
    
    def add_frame_change_post_handler(handlername,func):
        for handler in bpy.app.handlers.frame_change_post: 
            if handler.__name__ == handlername:
                return
            
        bpy.app.handlers.frame_change_post.append(func)
        return
        

    def remove_frame_change_post_handler(handlername):
        for handler in bpy.app.handlers.frame_change_post:
            if handler.__name__ == handlername:
                bpy.app.handlers.frame_change_post.remove(handler)
                return
            

class HUDTOGGLE_OT_Button(bpy.types.Operator):
    bl_idname = "hud_toggle.button"
    bl_label = "SHOW/HIDDEN"
    def execute(self,context):
        if HUD_View3d.ui:
            if RENDER_TEXT.blfHUD_show == True:
                RENDER_TEXT.blfHUD_show = False
                RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
                VariousCalculation.remove_frame_change_post_handler("stamp_entry")
                VariousCalculation.remove_frame_change_post_handler("get_previous_position")
                bpy.context.scene.render.stamp_note_text = ""
                RENDER_TEXT.redraw_regions()

            else:
                RENDER_TEXT.blfHUD_show = True
                VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                areasize = VariousCalculation.get_area_size()
                RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,HUD_View3d.ui),HUD_View3d.ui)
                RENDER_TEXT.redraw_regions()
                RENDER_TEXT.stamp_entry(bpy.context.scene)

            HUD_View3d.note_setting(HUD_View3d)

            return{'FINISHED'}
        
        else:
            self.app = QtWidgets.QApplication.instance()
            if not self.app:
                self.app = QtWidgets.QApplication(['blender'])
            self.event_loop = QEventLoop()
            oneshotGUI = HUD_View3d.createWindow(HUD_View3d)
            HUD_View3d.createWindow(HUD_View3d).showHUD()

            if RENDER_TEXT.blfHUD_show == True:
                RENDER_TEXT.blfHUD_show = False
                RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
                VariousCalculation.remove_frame_change_post_handler("stamp_entry")
                VariousCalculation.remove_frame_change_post_handler("get_previous_position")
                RENDER_TEXT.redraw_regions()
                HUD_View3d.note_setting(HUD_View3d)

            else:
                RENDER_TEXT.blfHUD_show = True
                VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                areasize = VariousCalculation.get_area_size()
                RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,oneshotGUI),oneshotGUI)
                RENDER_TEXT.redraw_regions()
                HUD_View3d.note_setting(HUD_View3d)
                RENDER_TEXT.stamp_entry(bpy.context.scene)
            return {'FINISHED'}


class SETUPDATE_OT_Button(bpy.types.Operator):
    bl_idname = "hud_setup.button"
    bl_label = "Set/Update HUD"

    def execute(self,context):
        if HUD_View3d.ui:
            if RENDER_TEXT.blfHUD_show == True:
                RENDER_TEXT.blfHUD_show = False
                try:
                    HUD_View3d.ui.showHUD()
                except:
                    pass
                RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
                RENDER_TEXT.redraw_regions()
                VariousCalculation.remove_frame_change_post_handler("stamp_entry")
                VariousCalculation.remove_frame_change_post_handler("get_previous_position")
                RENDER_TEXT.blfHUD_show = True
                areasize = VariousCalculation.get_area_size()
                VariousCalculation.update_HUD_SB(HUD_View3d.ui)
                RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,HUD_View3d.ui),
                                            HUD_View3d.ui)
                RENDER_TEXT.redraw_regions()
                VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                HUD_View3d.note_setting(HUD_View3d)
                RENDER_TEXT.stamp_entry(bpy.context.scene)
            else:
                RENDER_TEXT.blfHUD_show = True
                try:
                    HUD_View3d.ui.showHUD()
                except:
                    pass
                areasize = VariousCalculation.get_area_size()
                VariousCalculation.update_HUD_SB(HUD_View3d.ui)
                RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,HUD_View3d.ui),
                                            HUD_View3d.ui)
                RENDER_TEXT.redraw_regions()
                VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                HUD_View3d.note_setting(HUD_View3d)
                RENDER_TEXT.stamp_entry(bpy.context.scene)
        else:
            self.app = QtWidgets.QApplication.instance()
            if not self.app:
                self.app = QtWidgets.QApplication(['blender'])
            self.event_loop = QEventLoop()
            oneshotGUI = HUD_View3d.createWindow(HUD_View3d)
            HUD_View3d.createWindow(HUD_View3d).showHUD()

            try:
                if RENDER_TEXT.blfHUD_show == True:
                    RENDER_TEXT.blfHUD_show = False
                    oneshotGUI.showHUD()
                    RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
                    RENDER_TEXT.redraw_regions()
                    VariousCalculation.remove_frame_change_post_handler("stamp_entry")
                    VariousCalculation.remove_frame_change_post_handler("get_previous_position")
                    RENDER_TEXT.blfHUD_show = True
                    areasize = VariousCalculation.get_area_size()
                    VariousCalculation.update_HUD_SB(HUD_View3d.ui)
                    RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,HUD_View3d.ui),
                                                HUD_View3d.ui)
                    RENDER_TEXT.redraw_regions()
                    VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                    VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                    HUD_View3d.note_setting(HUD_View3d)
                    RENDER_TEXT.stamp_entry(bpy.context.scene)
                else:
                    RENDER_TEXT.blfHUD_show = True
                    oneshotGUI.showHUD()
                    areasize = VariousCalculation.get_area_size()
                    VariousCalculation.update_HUD_SB(HUD_View3d.ui)
                    RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",RENDER_TEXT.get_font_info(RENDER_TEXT,HUD_View3d.ui),
                                                HUD_View3d.ui)
                    RENDER_TEXT.redraw_regions()
                    VariousCalculation.add_frame_change_post_handler("stamp_entry",RENDER_TEXT.stamp_entry)
                    VariousCalculation.add_frame_change_post_handler("get_previous_position",VariousCalculation.get_previous_position)
                    HUD_View3d.note_setting(HUD_View3d)
                    RENDER_TEXT.stamp_entry(bpy.context.scene)
            except:
                pass

        return{'FINISHED'}

class HUDTOOLIMPORTJSON_OT_Button(bpy.types.Operator):
    bl_idname = "hudtools.import_jsonbutton"
    bl_label = "Import Settings"
    json_path = ""

    def execute(self,context):
        bpy.ops.hudtools.import_json('INVOKE_DEFAULT')
        return {'FINISHED'}
    

class HUDTOOLEXPORTJSON_OT_Button(bpy.types.Operator):
    bl_idname = "hudtools.export_jsonbutton"
    bl_label = "Export Settings"
    json_path = ""

    def execute(self,context):
        bpy.ops.hudtools.export_json('INVOKE_DEFAULT')
        return {'FINISHED'}

class HUD_View3d():
    ui = None
    def getHUDSize(i):
        return int(i)
    
    def note_setting(self):
        bpy.context.scene.render.metadata_input = 'SCENE'
        bpy.context.scene.render.use_stamp_date = False
        bpy.context.scene.render.use_stamp_time = False
        bpy.context.scene.render.use_stamp_render_time = False
        bpy.context.scene.render.use_stamp_frame = False
        bpy.context.scene.render.use_stamp_frame_range = False
        bpy.context.scene.render.use_stamp_memory = False
        bpy.context.scene.render.use_stamp_hostname = False
        bpy.context.scene.render.use_stamp_camera = False
        bpy.context.scene.render.use_stamp_lens = False
        bpy.context.scene.render.use_stamp_scene = False
        bpy.context.scene.render.use_stamp_marker = False
        bpy.context.scene.render.use_stamp_filename = False
        bpy.context.scene.render.use_stamp_sequencer_strip = False
        bpy.context.scene.render.use_stamp_note = True
        bpy.context.scene.render.use_stamp = True
        bpy.context.scene.render.stamp_font_size = RENDER_TEXT.fontSize
        bpy.context.scene.render.use_stamp_labels = False

    def createWindow(self):
        self.ui = GUI()
        wid = QApplication.topLevelWidgets()
        for i in wid:
            if self.ui.objectName() == i.objectName():
                i.close()

        self.ui.setAttribute(Qt.WA_DeleteOnClose)
        #self.ui.show()
        return self.ui

    def execute():
        if bpy.context.scene.hud_show == True:
            bpy.context.scene.hud_show = False
            bpy.app.handlers.frame_change_post.clear()
            objhandler = bpy.app.driver_namespace.get('draw_objectname')
            if objhandler:
                bpy.types.SpaceView3D.draw_handler_remove(objhandler,'WINDOW')
                del bpy.app.driver_namespace['draw_objectname']
                
            speedhandler = bpy.app.driver_namespace.get('draw_objectspeed')
            if speedhandler:
                bpy.types.SpaceView3D.draw_handler_remove(speedhandler,'WINDOW')
                del bpy.app.driver_namespace['draw_objectspeed']
            
            RENDER_TEXT.redraw_regions()

class GUI(QMainWindow, QFrame):
    buttonHeight = 16
    buttonWidth = 120
    blockNum = 12 #max:18
    defaultFileName = 'defaultSetting.json'
    defPath = '%s/%s'%(path, defaultFileName)
    baseDict = {}
    
    def __init__(self, *args, **kwargs):
        super(GUI, self).__init__(*args, **kwargs)
        
        self.setWindowTitle("HUD custom tools")
        self.setObjectName("HUDCustomToolsUI")
        
        self.initBaseDict()
        self.importSetting()

        #self.setWindowFlags(Qt.WindowStaysOnTopHint)

        self.setAcceptDrops(True)
        self.createMenu()
        self.createUI()
        
        # Default Settings import
        if os.path.exists(self.defPath): 
            self.reloadFromFile(self.defPath)
    

    def initBaseDict(self):
        sdict = {}
        for d in s.__dict__:
            try:
                if not '__' in d:
                    sdict[d] = s.__dict__[d]
            except:
                pass
    
        self.baseDict = copy.deepcopy(sdict)
        
    def createMenu(self):
        saveDefAct = QAction('Save As Default Settings', self)
        saveDefAct.triggered.connect(self.saveDef)
        initDefAct = QAction('Reset Settings', self)
        initDefAct.triggered.connect(self.initDef)
        visoffAct = QAction('All Visibility Off', self)
        visoffAct.triggered.connect(lambda: self.setVis(False))
        visonAct = QAction('All Visibility On', self)
        visonAct.triggered.connect(lambda: self.setVis(True))

        
        menuBar = self.menuBar()
        filemenu = menuBar.addMenu('File')
        filemenu.addSeparator()
        filemenu.addAction(saveDefAct)
        filemenu.addAction(initDefAct)
        editmenu = menuBar.addMenu('Edit')
        editmenu.addAction(visonAct)
        editmenu.addAction(visoffAct)

    
    def saveDef(self):
        output = self.getExportJson()
        with open('%s/%s'%(path, self.defaultFileName), mode = 'w') as f:
            json.dump(output, f, indent=4)
                    
    def initDef(self):
        self.initBaseDict()
        if os.path.exists(self.defPath):
            ret = QMessageBox.information(
                None,
                'Confirm',
                'Delete default settings?',
                QMessageBox.Yes,
                QMessageBox.No
            )
            if ret == QMessageBox.Yes:
                os.remove(self.defPath)
        
        camSpeedCBIndex = self.camSpeedUnitCB.findText(camSpeedUnit)
        velSpeedCBIndex = self.velSpeedUnitCB.findText(velSpeedUnit)
        self.camSpeedUnitCB.setCurrentIndex(camSpeedCBIndex)
        self.velSpeedUnitCB.setCurrentIndex(velSpeedCBIndex)
        self.distanceCB.setCurrentIndex(self.distanceCB.findText(distanceUnit))
        self.fontsizeCB.setValue(self.HUDSize)
        
        for key in self.baseDict:
            val  = self.baseDict[key]
            if type(val) == dict:
                for i in range(5):
                    if val.get('name') == 'HUDDistance%s'%(i+1):
                        _t = val.get('node')
                        _l = val.get('label')
                        _l = _l.replace(' : ','')
                        self.distanceNameLEs[i].setText(_t)
                        self.distanceLabelLEs[i].setText(_l)
                    if val.get('name') == 'HUDVel%s'%(i+1):
                        _t = val.get('node')
                        _l = val.get('label')
                        _l = _l.replace(' : ','')
                        self.velocityNameLEs[i].setText(_t)
                        self.velocityLabelLEs[i].setText(_l)
                for i in range(3):
                    if val.get('name') == 'HUDComment%s'%(i+1):
                        _t = val.get('label')
                        self.commentLEs[i].setText(_t)
            
        self.reloadButtonLabels()
        
    def setVis(self, on):
        for b in self.buttons:
            b.setVis(on)

    
    def reloadFromFile(self, filePath):
        with open(filePath, mode = 'r') as f:
            imports = json.load(f)
        for dKey in self.baseDict:
            d = self.baseDict[dKey]
            if type(d) == dict:
                for importName in imports:
                    if d.get('name') == importName:
                        importSettings = imports[importName]
                        for k in importSettings:
                            self.baseDict[dKey][k] = importSettings[k]
                        
                        for i in range(5):
                            if importName == 'HUDVel%s'%(i+1):
                                _t = importSettings.get('node')
                                _l = importSettings.get('label')
                                _l = _l.replace(' : ','')
                                self.velocityNameLEs[i].setText(_t)
                                self.velocityLabelLEs[i].setText(_l)
                            if importName == 'HUDDistance%s'%(i+1):
                                _t = importSettings.get('node')
                                _l = importSettings.get('label')
                                _l = _l.replace(' : ','')
                                self.distanceNameLEs[i].setText(_t)
                                self.distanceLabelLEs[i].setText(_l)
                        for i in range(3):
                            if importName == 'HUDComment%s'%(i+1):
                                _t = importSettings.get('label')
                                self.commentLEs[i].setText(_t)

        if 'unitOfCamSpeed' in imports:
            self.camSpeedUnitCB.setCurrentText(imports['unitOfCamSpeed'])
        if 'unitOfVelSpeed' in imports:
            self.velSpeedUnitCB.setCurrentText(imports['unitOfVelSpeed'])
        if 'unitOfDistance' in imports:
            self.distanceCB.setCurrentText(imports['unitOfDistance'])
        if 'HUDSizeNum' in imports:
            self.fontsizeCB.setValue(imports['HUDSizeNum'])
            
        self.reloadButtonLabels()


    def getExportJson(self):
        output = {}
        output['unitOfCamSpeed'] = self.camSpeedUnitCB.currentText()
        output['unitOfVelSpeed'] = self.velSpeedUnitCB.currentText()
        output['unitOfDistance'] = self.distanceCB.currentText()
        output['HUDSizeNum'] = self.fontsizeCB.value()
        for b in self.buttons:
            nam = b.setting.nam
            if nam: 
                output[nam] =  {
                    'section': b.setting.section, 
                    'block' : b.setting.block,
                    'vis' : b.setting.vis,
                    'node' : b.setting.node,
                    'label' : b.setting.lab,
                }
        return output
            
    def dragEnterEvent(self, event):
        super(GUI, self).dragEnterEvent(event)
        event.accept()
        
    def dragLeaveEvent(self, event):
        super(GUI, self).dragLeaveEvent(event)

        for hud in self.buttons:
            hud.releaseDrag()
        event.accept()
        
    def isLabelEnter(self):
        for hud in self.buttons:
            if hud.isEnter:
                return True
        return False
        
    def dropEvent(self, event):
        super(GUI, self).dropEvent(event)
        for hud in self.buttons:
            hud.releaseDrag()
        event.accept()

    def importSetting(self):
        self.dictSB={}
        self.HUDSize = 16
        
        doubleKey = []
        for setting in self.baseDict.keys():
            key = self.baseDict[setting]
            if type(key) == dict:
                if 'section' in key:
                    keyT = (key['section'], key['block'])
                    if keyT in self.dictSB:
                        doubleKey.append(self.dictSB[keyT])
                    self.dictSB[keyT] = setting
            else:
                if setting == 'HUDSizeNum': self.HUDSize = HUD_View3d.getHUDSize(key)
        for dk in doubleKey:
            for sec in range(10):
                for blk in range(self.blockNum):
                    keyT = (sec, blk)
                    if not keyT in self.dictSB:
                        self.dictSB[keyT] = dk
                        self.baseDict[dk]['section'] = sec
                        self.baseDict[dk]['block'] = blk
                        break
                else: continue
                break

        global camSpeedUnit
        camSpeedUnit = self.baseDict['unitOfCamSpeed']
        global velSpeedUnit
        velSpeedUnit = self.baseDict['unitOfVelSpeed']
        global distanceUnit
        distanceUnit = self.baseDict['unitOfDistance']

    def createUI(self):
        cw = QWidget()
        self.setCentralWidget(cw)
        cw.setStyleSheet("QWidget{background-color: rgba(48,48,48,1);}")
        
        lo = QVBoxLayout(cw)
        loU = QHBoxLayout()
        loD = QHBoxLayout()

        l = QLabel('-- HUD custom tools for Blender v1.0.0 --')
        l.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(220,104,0,1);}")
        l.setFixedHeight(20)

        lo.addWidget(l)

        #buttons UI
        loUDFL = QHBoxLayout()
        loUDF = QmyFrame()
        
        loUDF.setFrameStyle(QFrame.Panel | QFrame.Raised)
        loUDF.setLineWidth(2)
        loUDF.setStyleSheet(
            "QmyFrame{background-image : url('%s/backgroundGUI.png')}"%path
        )
        loUD = QVBoxLayout()
        
        #loUDF.setFixedSize(740,(314*2))
        loUDF.setFixedSize(740,314)
        
        lo.addLayout(loUDFL)
        loUDFL.addWidget(loUDF)
        loUDF.setLayout(loUD)
        loUD.addSpacing(30)

        self.buttons=[]
        loUD.addLayout(loU)
        loU.addSpacing(30)
        for sec in range(5):
            loSection = QVBoxLayout()
            loSection.addSpacing(10)
            loU.addLayout(loSection)
            for blc in range(self.blockNum):
                dict,label = self.getButtonLabel(sec, blc)
                lab = DraggableLabel(label, setting=dict)
                lab.setFixedSize(self.buttonWidth, self.buttonHeight)
                loSection.addWidget(lab)
                self.buttons.append(lab)
        loM=QVBoxLayout()
        loUD.addLayout(loM)
        loM.addSpacing(10)
        #separate
        sep = QFrame()
        sep.setFrameShape( QFrame.HLine )
        sep.setFrameShadow( QFrame.Raised )
        sep.setSizePolicy(
            QSizePolicy.MinimumExpanding,
            QSizePolicy.MinimumExpanding
        )
        lo.addWidget(sep)

        f = QFrame()
        f.setStyleSheet("QFrame{background-color: rgba(48,48,48,1)}")
        
        f.setFixedHeight(230)
        loS1 = QVBoxLayout()
        loS2 = QHBoxLayout()

        loS1.addLayout(loS2)

        # user name
        loS2.addStretch(27)
        l = QLabel('User Name:')
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        
        l.setFixedWidth(70)
        loS2.addWidget(l)

        self.userLE = QLineEdit(eval(self.baseDict['userSettings']['command']))
        self.userLE.setFixedWidth(110)
        self.userLE.textEdited.connect(lambda: self.changeUsername())
        self.userLE.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
        
        loS2.addWidget(self.userLE)
        loS2.addStretch(25)

        # unit of camSpeed
        l = QLabel('Unit of Velocity(Camera):')
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        l.setFixedWidth(160)
        loS2.addWidget(l)
        
        self.camSpeedUnitCB = QComboBox()
        self.camSpeedUnitCB.setStyleSheet\
            ("QComboBox{color: rgba(200,200,200,1); \
             background-color: rgba(40,40,40,1);border: none;}\
             QComboBox QAbstractItemView{color: rgba(255,255,255,1); \
             background-color: rgba(60,60,60,1);}")
        self.camSpeedUnitCB.addItem('m/s ')
        self.camSpeedUnitCB.addItem('km/h')
        self.camSpeedUnitCB.addItem('mph ')
        self.camSpeedUnitCB.setFixedWidth(70)
        camSpeedunitCBIndex = self.camSpeedUnitCB.findText(camSpeedUnit)
        self.camSpeedUnitCB.setCurrentIndex(camSpeedunitCBIndex)
        self.camSpeedUnitCB.currentIndexChanged.connect(lambda: self.changeCamVelUnit())
        loS2.addWidget(self.camSpeedUnitCB)
        loS2.addStretch(33)        
                
                
        # font size
        l = QLabel('Font Size:')
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        l.setFixedWidth(60)
        loS2.addWidget(l)
        
        self.fontsizeCB = QSpinBox()
        self.fontsizeCB.setStyleSheet("QSpinBox{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
        self.fontsizeCB.setMinimum(8)
        self.fontsizeCB.setMaximum(64)
        self.fontsizeCB.setFixedWidth(90)
        self.fontsizeCB.setValue(self.HUDSize)
        loS2.addWidget(self.fontsizeCB)
        loS2.addStretch(31)

        self.fontsizeCB.valueChanged.connect(self.changeFontSize)
        
        # comments
        loS6 = QHBoxLayout()
        loS1.addLayout(loS6)
        comment1Lb = QLabel('Comment1: ')
        comment2Lb = QLabel('Comment2: ')
        comment3Lb = QLabel('Comment3: ')
        comment1Lb.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        comment2Lb.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        comment3Lb.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        comment1Lb.setFixedWidth(70)
        comment2Lb.setFixedWidth(70)
        comment3Lb.setFixedWidth(70)
        self.commentLEs = []
        l = QLineEdit()
        l.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 0:self.changeComment(_))
        self.commentLEs.append(l)
        l = QLineEdit()
        l.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 1:self.changeComment(_))
        self.commentLEs.append(l)
        l = QLineEdit()
        l.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
        l.setFixedWidth(110)
        l.textChanged.connect(lambda s, _ = 2:self.changeComment(_))
        self.commentLEs.append(l)

        
        loS6.addStretch(27)
        loS6.addWidget(comment1Lb)
        loS6.addWidget(self.commentLEs[0])
        loS6.addStretch(25)
        loS6.addWidget(comment2Lb)
        loS6.addWidget(self.commentLEs[1])
        loS6.addStretch(33)
        loS6.addWidget(comment3Lb)
        loS6.addWidget(self.commentLEs[2])
        loS6.addStretch(31)

        sep = QFrame()
        sep.setFrameShape( QFrame.HLine )
        sep.setFrameShadow( QFrame.Raised )
        sep.setSizePolicy(
            QSizePolicy.MinimumExpanding,
            QSizePolicy.MinimumExpanding
        )
        loS1.addWidget(sep)
        
        loS4 = QHBoxLayout()
        loS1.addLayout(loS4)
        
        loS3 = QVBoxLayout()
        
        loS4.addWidget(QLabel(''))
        loS4.addSpacing(10)
        loS4.addLayout(loS3)
        
        loS5 = QHBoxLayout()
        loS3.addLayout(loS5)
        
        # unit of velSpeed        
        l = QLabel('Unit of Velocity(Vel1-5):')
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        
        l.setFixedWidth(170)
        loS5.addWidget(l)
        
        self.velSpeedUnitCB = QComboBox()
        self.velSpeedUnitCB.setStyleSheet\
            ("QComboBox{color: rgba(200,200,200,1); \
             background-color: rgba(40,40,40,1);border: none;}\
             QComboBox QAbstractItemView{color: rgba(255,255,255,1); \
             background-color: rgba(60,60,60,1);}")
        self.velSpeedUnitCB.addItem('m/s ')
        self.velSpeedUnitCB.addItem('km/h')
        self.velSpeedUnitCB.addItem('mph ')
        self.velSpeedUnitCB.setFixedWidth(70)
        velSpeedUnitCBIndex = self.velSpeedUnitCB.findText(velSpeedUnit)
        self.velSpeedUnitCB.setCurrentIndex(velSpeedUnitCBIndex)
        loS5.addWidget(self.velSpeedUnitCB)
        loS5.addSpacing(50)
        
        self.velocityNameLEs = []
        self.velocityLabelLEs = []
        self.setVelocityPBs = []
        
        for i in range(5):
        # node vel
            loS5 = QHBoxLayout()
            l = QLabel('Velocity%s Object:'%(i+1))
            l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
            l.setFixedWidth(130)
            loS5.addWidget(l)

            l = QLineEdit()
            l.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
            loS5.addWidget(l)
            l.textChanged.connect(lambda _,__=i: self.changeVelocityName(__))
            l.setFixedWidth(120)
            self.velocityNameLEs.append(l)

            l = QmyPushButton(self, '<<', 'velocity', i, width=20, layout=loS5)

            self.setVelocityPBs.append(l)
            
            vsLabel = self.baseDict['vel%sSettings'%(i+1)]['label']
            l=QLineEdit(vsLabel.replace(' : ',''))
            self.velocityLabelLEs.append(l)
            loS3.addLayout(loS5)
        
        
        loS4.addSpacing(100)
        loS3 = QVBoxLayout()
        
        loS4.addWidget(QLabel(''))
        loS4.addLayout(loS3)
        
        loS5 = QHBoxLayout()
        loS3.addLayout(loS5)
        
        # unit of distance        
        l = QLabel('Unit of Distance(Dist1-5):')
        l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
        l.setFixedWidth(180)
        loS5.addWidget(l)
        
        self.distanceCB = QComboBox()
        self.distanceCB.setStyleSheet\
            ("QComboBox{color: rgba(200,200,200,1); \
             background-color: rgba(40,40,40,1);border: none;}\
             QComboBox QAbstractItemView{color: rgba(255,255,255,1); \
             background-color: rgba(60,60,60,1);}")
        self.distanceCB.addItem('m ')
        self.distanceCB.addItem('ft')
        self.distanceCB.setFixedWidth(70)
        self.distanceCB.setCurrentIndex(self.distanceCB.findText(distanceUnit))
        loS5.addWidget(self.distanceCB)
        loS5.addSpacing(50)
        
        self.distanceNameLEs = []
        self.distanceLabelLEs = []
        self.setDistancePBs = []
        
        ##distance
        for i in range(5):
            loS5 = QHBoxLayout()
            l = QLabel('Distance%s Object:'%(i+1))
            l.setStyleSheet("QLabel{color: rgba(230,230,230,1)}")
            l.setFixedWidth(130)
            loS5.addWidget(l)

            l = QLineEdit()
            l.setStyleSheet("QLineEdit{color: rgba(200,200,200,1); background-color: rgba(40,40,40,1);border: none;}")
            loS5.addWidget(l)
            l.textChanged.connect(lambda s, _ = i:self.changeDistanceName(_))
            l.setFixedWidth(120)
            self.distanceNameLEs.append(l)
            
            l = QmyPushButton(self, '<<', 'distance', i, width=20, layout=loS5)
            self.setDistancePBs.append(l)
            
            dsLabel = self.baseDict['distance%sSettings'%(i+1)]['label']
            l = QLineEdit(dsLabel.replace(' : ',''))
            self.distanceLabelLEs.append(l)
            
            loS3.addLayout(loS5)
        
        loS4.addWidget(QLabel(''))
        loS4.addWidget(QLabel(''))
        
        f.setLayout(loS1)
        lo.addWidget(f)
        lo.addSpacing(10)

        loB = QHBoxLayout()
        self.getstamplabel = QLabel()
        self.getstamplabel.setText("Note Limit:  "+"0/511")
        self.getstamplabel.setAlignment(Qt.AlignHCenter | Qt.AlignVCenter)
        #self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,104,0,1);}")
        self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")
        self.getstamplabel.setFixedWidth(180)
        self.getstamplabel.setFixedHeight(20)
        loB.addWidget(self.getstamplabel)
        loB.setAlignment(Qt.AlignHCenter)
        lo.addLayout(loB)

    def setLw(self, b, label):
        length = len(label)
        b.setting.lw = 100
        if length > 13:
            b.setting.lw = length*8



    def setVelocityName(self, i):
        if self.velocityNameLEs[i].text() == '':
            sl = bpy.context.view_layer.objects.active
            if not sl:
                return
            self.velocityNameLEs[i].setText(sl.name_full)
        else:
            self.velocityNameLEs[i].setText('')
        pass
    
    def setDistanceName(self, i):
        if self.distanceNameLEs[i].text() == '':
            sl = bpy.context.view_layer.objects.active
            if not sl:
                return
            self.distanceNameLEs[i].setText(sl.name_full)
        else:
            self.distanceNameLEs[i].setText('')
        pass
                
    
    def changeFontSize(self,value):
        RENDER_TEXT.fontSize = self.fontsizeCB.value()

    def changeCamVelUnit(self):
        camSpeedUnit = self.camSpeedUnitCB.currentText()
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")

    def changeDistanceName(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDDistance%s'%(i+1):
                b.setting.node = self.distanceNameLEs[i].text()
                lab = 'Dist%s %s'%(i+1, b.setting.node)
                
                b.setVis(0 if  self.distanceNameLEs[i].text() == '' else 1)
                b.setting.lab = '%s : '%lab
                b.setText(lab)
                self.setLw(b, b.setting.lab)
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")


        
    def changeDistanceLabel(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDDistance%s'%(i+1):
                b.setting.lab = "%s : "%self.distanceLabelLEs[i].text()
                b.setText(self.distanceLabelLEs[i].text())
        
                self.setLw(b, b.setting.lab)
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")


                
    def changeUsername(self):
        for b in self.buttons:
            if b.setting.nam == 'HUDUser':
                b.setting.cmd = "'%s'"%self.userLE.text()
                b.setText(self.userLE.text())
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")


    def changeComment(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDComment%s'%(i+1):
                b.setting.lab = '%s'%self.commentLEs[i].text()
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")



    def changeVelocityName(self,i):
        for b in self.buttons:
            if b.setting.nam == 'HUDVel%s'%(i+1):
                b.setting.node = self.velocityNameLEs[i].text()
                lab = 'Vel%s %s'%(i+1, b.setting.node)
                
                b.setVis(0 if  self.velocityNameLEs[i].text() == '' else 1)
                b.setting.lab = '%s : '%lab
                b.setText(lab)
                self.setLw(b, b.setting.lab)
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")


                
    def changeVelocityLabel(self, i):
        for b in self.buttons:
            if b.setting.nam == 'HUDVel%s'%(i+1):
                b.setting.lab = "%s : "%self.velocityLabelLEs[i].text()
                b.setText(self.velocityLabelLEs[i].text())
        
                self.setLw(b, b.setting.lab)
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")



    def showHUD(self):
        global camSpeedUnit
        camSpeedUnit = self.camSpeedUnitCB.currentText()
        global velSpeedUnit
        velSpeedUnit = self.velSpeedUnitCB.currentText()
        global distanceUnit
        distanceUnit = self.distanceCB.currentText()
        global distanceNodes
        global velNodes
        for i in range(5):
            distanceNodes[i] = self.distanceNameLEs[i].text()
            velNodes[i] = self.velocityNameLEs[i].text()

        for b in self.buttons:
            b.setting.size = self.fontsizeCB.value()
            b.setting.setHUD()
        RENDER_TEXT.fontSize = self.fontsizeCB.value()
        note_length = str(VariousCalculation.get_note_length(self))
        self.getstamplabel.setText("Note Limit:  "+note_length+"/511")
        if VariousCalculation.get_note_length(self) >511:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
        else:
            self.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")




    def getButtonLabel(self, sec, blc):
        dict = {}
        name = ''
        label = ''
        dict['section'] = sec
        dict['block'] = blc
        if (sec, blc) in self.dictSB:
            dict = self.baseDict[self.dictSB[(sec, blc)]]
            name = dict['name'].replace('HUD','')
            try :
                label = eval( dict['command'] )
            except:
                pass
            
            if 'label' in dict:
                label = dict['label'].replace(':','')
            if not label:
                label = name
            
            if label == 'Time':
                label = eval( dict['command'] )
            if name == 'CamName':
                label = 'CamName'
            for i in range(1,4):
                if name == 'Comment%s'%i:
                    label = 'Comment%s'%i
            
        dict['size'] = self.HUDSize
        return dict,label
    
    def reloadButtonLabels(self):
        self.importSetting()
        for sec in range(5):
            for blk in range(self.blockNum):
                blk_ = blk if sec < 5 else self.blockNum-blk-1
                dict,label = self.getButtonLabel(sec, blk_)
                self.buttons[sec*self.blockNum + blk].setting.reset(dict)
                self.buttons[sec*self.blockNum + blk].setText(label)
                self.buttons[sec*self.blockNum + blk].setVis(
                    self.buttons[sec*self.blockNum + blk].setting.vis
                )
        self.changeUsername()
        for i in range(5):
            self.changeDistanceLabel(i)
            self.changeVelocityLabel(i)
            

class PYSIDE_OT_display_window(Operator):
    bl_idname = 'hudtools.show_gui'
    bl_label = "Open Setting GUI"
    bl_options = {'REGISTER'}
    ui = None

    def createHUDWindow(self):
        ui = GUI()
        wid = QApplication.topLevelWidgets()
        for i in wid:
            if ui.objectName() == i.objectName():
                i.close()

        ui.setAttribute(Qt.WA_DeleteOnClose)
        #ui.setWindowFlags(Qt.WindowStaysOnTopHint)
        ui.show()
        return ui
    
    def execute(self, context):
        
        self.app = QtWidgets.QApplication.instance()
        if not self.app:
            self.app = QtWidgets.QApplication(['blender'])

        self.event_loop = QEventLoop()
        self.widget = HUD_View3d.createWindow(HUD_View3d)
        self.widget.show()
        areasize = VariousCalculation.get_area_size()
        #VariousCalculation.update_HUD_SB(self.widget)
        #if RENDER_TEXT.blfHUD_show == True:
        #    RENDER_TEXT.blfHUD_show = False
        #    RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
        #    RENDER_TEXT.redraw_regions()
        #    RENDER_TEXT.blfHUD_show = True
        #    RENDER_TEXT.add_driver_handler(RENDER_TEXT,areasize[0],areasize[1],"draw_HUD_View3d",
        #                                   RENDER_TEXT.get_font_info(RENDER_TEXT,self.widget),self.widget)
        #    RENDER_TEXT.redraw_regions()
        #else:
        #    pass

        return {'FINISHED'}


class ImportJsonData(Operator, ImportHelper):
    bl_idname = "hudtools.import_json"
    bl_label = "Import Settings"

    filename_ext = ".json"

    filter_glob: StringProperty(
        default="*.json",
        options={'HIDDEN'},
        maxlen=255,
    )

    def execute(self, context):
        if self.filepath == None:
            return False
        if HUD_View3d.ui:
            try:
                HUD_View3d.ui.reloadFromFile(self.filepath)
            except:
                ShowMessageBox("Use with hudtool GUI displayed.", "HUDTool Warning", 'ERROR')
        else:
            ShowMessageBox("Use with hudtool GUI displayed.", "HUDTool Warning", 'ERROR')
            pass
        return {'FINISHED'}
    

class ExportJsonData(Operator, ExportHelper):
    bl_idname = "hudtools.export_json"
    bl_label = "Export Settings"
    filename_ext = ".json"

    filter_glob: StringProperty(
        default="*.json",
        options={'HIDDEN'},
        maxlen=255,
    )

    def execute(self, context):
        if self.filepath == None:
            return False
        if HUD_View3d.ui:
            try:
                output = HUD_View3d.ui.getExportJson()
                with open(self.filepath, mode='w') as f:
                    json.dump(output, f, indent=4)
            except:
                ShowMessageBox("Use with hudtool GUI displayed.", "HUDTool Warning", 'ERROR')
        else:
            ShowMessageBox("Use with hudtool GUI displayed.", "HUDTool Warning", 'ERROR')
            pass

        return {'FINISHED'}


classes = (
    HUDTOOL_PT_Panel,
    HUDTOGGLE_OT_Button,
    HUDTOOLIMPORTJSON_OT_Button,
    HUDTOOLEXPORTJSON_OT_Button,
    SETUPDATE_OT_Button,
    PYSIDE_OT_display_window,
    ImportJsonData,
    ExportJsonData,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    RENDER_TEXT.add_load_pre_handler("draw_HUD_View3d")
    regist_props()
    print("register")
    if enable_Note_on_Save_pre_handler not in bpy.app.handlers.save_pre:
        bpy.app.handlers.save_pre.append(enable_Note_on_Save_pre_handler)
        print("Add note Pre Save handler.")
        print(bpy.app.handlers.save_pre)
    else:
        print("error")
        print(bpy.app.handlers.save_pre)
    if enable_Note_on_Save_post_handler not in bpy.app.handlers.save_post:
        bpy.app.handlers.save_post.append(enable_Note_on_Save_post_handler)
        print("Add note Post Save handler.")
        print(bpy.app.handlers.save_post)
        
    else:
        print("error")
        print(bpy.app.handlers.save_post)
    
def unregister():
    unregist_props()
    for cls in classes:
        bpy.utils.unregister_class(cls)
    VariousCalculation.remove_frame_change_post_handler("stamp_entry")
    VariousCalculation.remove_frame_change_post_handler("get_previous_position")
    RENDER_TEXT.del_load_pre_handler("draw_HUD_View3d")
    RENDER_TEXT.delete_driver_handler("draw_HUD_View3d")
    print("unregister")
    if enable_Note_on_Save_pre_handler in bpy.app.handlers.save_pre:
        bpy.app.handlers.save_pre.remove(enable_Note_on_Save_pre_handler)
    if enable_Note_on_Save_post_handler in bpy.app.handlers.save_post:
        bpy.app.handlers.save_post.remove(enable_Note_on_Save_post_handler)


class HUDsetting():
    def __init__(self,dict):
        self.reset(dict)
    def reset(self,dict):
        self.section = dict.get('section', '')
        self.block = dict.get('block', '')
        self.nam = dict.get('name', '')
        self.lab = dict.get('label', '')
        self.cmd = dict.get('command', '')
        self.evt = dict.get('event', '')
        self.lw = dict.get('lw', 80)
        self.dw = dict.get('dw', 30)
        self.ba = dict.get('ba', 'left')
        self.da = dict.get('da', 'left')
        self.size = dict.get('size', 16)
        self.vis = dict.get('vis', 1 if self.nam else 0)
        self.node = dict.get('node','')
 
    def setHUD(self):    
        kwargs = {
            'section' : self.section, 
            'block' : self.block,
            'labelWidth' : self.lw,
            'dataWidth' : self.dw,
            'label' : self.lab,
            'blockAlignment' : self.ba,
            'dataAlignment' : self.da,
            'labelFontSize' : self.size,
            'dataFontSize' : self.size,
        }


class DraggableLabel(QLabel):
    isEnter = False
    startPos = QPoint(0, 0)
    def __init__(self, *args, **kwargs):
        dict = kwargs.pop('setting')
        self.setting = HUDsetting(dict)
        
        super(DraggableLabel, self).__init__(*args, **kwargs)
        
        self.setAcceptDrops(True)
        self.backgroundColor(0)
        self.drag = None

    def mousePressEvent(self, event):
        super(DraggableLabel, self).mousePressEvent(event)
        if event.button() == Qt.LeftButton:
            self.startPos = event.pos()
            event.accept()

    def mouseMoveEvent(self, event):
        distance = (event.pos() - self.startPos).manhattanLength()
        if distance > QApplication.startDragDistance():
            mimeData = QMimeData()
            mimeData.setText(self.text())
            
            self.backgroundColor(2)
            
            self.drag = QDrag(self)
            self.drag.setMimeData(mimeData)
            self.drag.exec_()
            
        event.accept()

    def dropEvent(self, event):
        super(DraggableLabel, self).dropEvent(event)
        if event.mimeData().hasText():
            event.source().backgroundColor(0)
            event.source().drag = None
            
            #flip text
            tmp = event.source().text()
            event.source().setText(self.text())
            self.setText(tmp)

            #flip setting
            tmp = event.source().setting
            event.source().setting = self.setting
            self.setting = tmp

            #nonflip section,block
            tmp = event.source().setting.section
            event.source().setting.section = self.setting.section
            self.setting.section = tmp
            tmp = event.source().setting.block
            event.source().setting.block = self.setting.block
            self.setting.block = tmp
            
            event.source().backgroundColor(0)
            
        self.backgroundColor(0)
        self.drag = None
        event.accept()
        try:
            note_length = str(VariousCalculation.get_note_length(HUD_View3d.ui))
            HUD_View3d.ui.getstamplabel.setText("Note Limit:  "+note_length+"/511")
            if VariousCalculation.get_note_length(self) >511:
                HUD_View3d.ui.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
            else:
                HUD_View3d.ui.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")

        except:
            pass
    
    def setVis(self, on):
        if self.setting.nam:
            self.setting.vis = on
        self.backgroundColor(0)
        try:
            note_length = str(VariousCalculation.get_note_length(HUD_View3d.ui))
            HUD_View3d.ui.getstamplabel.setText("Note Limit:  "+note_length+"/511")
            if VariousCalculation.get_note_length(HUD_View3d.ui) >511:
                HUD_View3d.ui.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(200,30,30,1);}")
            else:
                HUD_View3d.ui.getstamplabel.setStyleSheet("QLabel{color: rgba(230,230,230,1); background-color: rgba(60,60,60,1);}")
        except:
            pass
        
    def mouseReleaseEvent(self, event):
        super(DraggableLabel, self).mouseReleaseEvent(event)
        if self.setting.nam:
            self.setVis(1 - self.setting.vis)
        event.accept()

    def releaseDrag(self):
        self.backgroundColor(0)
        self.drag = None

    def dragLeaveEvent(self, event):
        super(DraggableLabel, self).dragLeaveEvent(event)
        if not self.drag:
            self.backgroundColor(0)

        event.accept()
        
    def dragEnterEvent(self, event):
        super(DraggableLabel, self).dragEnterEvent(event)
        if type(event.source()) == type(self):
            if not self.drag:
                self.backgroundColor(1)
                event.source().backgroundColor(2)
            event.accept()
        else:
            event.ignore()

    def backgroundColor(self,i):
        alpha = 0.75 if self.setting.vis else 0.25
        colors = [
            #main
            'rgba(214,98,0,%s)'%alpha,
            #drag to
            'rgba(250,134,0,%s)'%alpha,
            #drag from
            'rgba(38,87,135,%s)'%alpha,
            #Highlight
            'rgba(255,144,0,%s)'%alpha
        ]
        self.setStyleSheet('QLabel {color: rgba(200,200,200,1); background-color :%s}'%colors[i])
        
    
    def enterEvent(self, event):
        super(DraggableLabel, self).enterEvent(event)
        isEnter = True
        self.backgroundColor(3)
        event.accept()
    
    def leaveEvent(self, event):
        super(DraggableLabel, self).enterEvent(event)
        isEnter = False
        if not self.drag:
            self.backgroundColor(0)
        event.accept()

def get_fps():
    render = bpy.context.scene.render
    scene_fps = render.fps / render.fps_base
    scene_fps = round(scene_fps,2)
    if scene_fps.is_integer():
        scene_fps = '{:<5d} {}'.format(int(scene_fps), "fps")
    else:
        scene_fps = '{:<5.2f} {}'.format(scene_fps, "fps")
    return scene_fps

def getNodeVel(i):
    node = velNodes[i-1]
    return getVel(node, velSpeedUnit)
    
def getCamVel():
    return getVel(VariousCalculation.get_scene_camera(), camSpeedUnit)
    
def getVel(node, unit):
    try:
        if VariousCalculation().check_exists_object(node) == True:
            distance = str(list(filter(lambda x: x[0] == str(node), objVelocities))[0][1])
        else: return '{:>7.3} {}'.format('---------', unit)
    except:
        return '{:>7.3} {}'.format('---------', unit)
    distanceUnit, timeUnit = {
        'mph ' : ('mi', 2.237), 
        'km/h' : ('km', 60*60*0.001),
        'm/s ' : ('m', 1)
    }[unit]
    if distance.endswith(distanceUnit):
        distance = distance.replace(distanceUnit, '')

    distance = float(distance)
    vel = abs(distance * bpy.context.scene.render.fps * timeUnit)
    return '{:>7.3f} {}'.format(vel, unit)

def getDistance(i):
    i-=1
    if VariousCalculation().check_exists_object(distanceNodes[i]) == True:
        for sceneobj in bpy.context.scene.objects:
            if distanceNodes[i] == sceneobj.name_full:
                obj = sceneobj
        #obj = bpy.data.objects[distanceNodes[i]]
        objpos = obj.matrix_world.translation
        pass
    else:
        return '{:>9.3} {}'.format('---------', distanceUnit)
    if bpy.context.scene.camera == None:
        return '{:>9.3} {}'.format('---------', distanceUnit)
    else:
        camera = bpy.context.scene.camera
        campos = camera.matrix_world.translation

    baseDistance = (campos - objpos).length * bpy.context.scene.unit_settings.scale_length
    if distanceUnit == "m ":
        pass
    elif distanceUnit == "ft":
        baseDistance = baseDistance * 3.28084
    else:
        pass

    if str(baseDistance).endswith(distanceUnit):
        baseDistance = baseDistance.replace(distanceUnit,'')
    norm = float(baseDistance)
    #
    return '{:>9.3f} {}'.format(norm, distanceUnit)
    #return distance
    pass


    
class QmyFrame(QFrame):
    def __init__ (self,*args,**kwargs):
        super(QmyFrame, self).__init__(*args, **kwargs)


class QmyPushButton():
    def __init__(self, gui, text, VD, i, width, layout):
        self.num = i
        self.gui = gui
        self.button = QPushButton(text)
        self.button.setStyleSheet("QPushButton{color: rgba(230,230,230,1);}")
        self.button.setFixedWidth(width)
        layout.addWidget(self.button)
        self.VD = VD
        self.button.clicked.connect(self.clicked)
        
    def clicked(self):
        if(self.VD == 'velocity'):
            self.gui.setVelocityName(self.num)
            
        if(self.VD == 'distance'):
            self.gui.setDistanceName(self.num)

if __name__ == "__main__":
    register()